<?php

namespace App\Services;

use App\Models\Transfer;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class WalletService
{
    public function transfer(User $from, string $toPhone, float $amount, ?string $note, ?string $idempotencyKey = null): Transfer
    {
        $toPhone = preg_replace('/\D/','',$toPhone);
        $to = User::where('phone', $toPhone)->first();

        if (!$to || $to->id === $from->id) {
            throw ValidationException::withMessages(['to_phone' => 'Receiver not found or invalid.']);
        }

        return DB::transaction(function () use ($from, $to, $amount, $note, $idempotencyKey) {

            // Idempotency
            if ($idempotencyKey) {
                $existing = Transfer::where('idempotency_key',$idempotencyKey)
                    ->where('from_user_id',$from->id)
                    ->where('to_user_id',$to->id)
                    ->where('amount',$amount)
                    ->first();
                if ($existing) return $existing;
            }

            // Lock rows in order to avoid deadlocks
            $ids = collect([$from->id, $to->id])->sort()->values();
            DB::table('users')->whereIn('id', $ids)->lockForUpdate()->get();

            $from->refresh();
            $to->refresh();

            if ($from->balance < $amount) {
                throw ValidationException::withMessages(['amount' => 'Insufficient balance.']);
            }

            // Update balances
            $from->balance -= $amount;
            $to->balance   += $amount;
            $from->save();
            $to->save();

            // Create transfer record
            $transfer = Transfer::create([
                'from_user_id' => $from->id,
                'to_user_id'   => $to->id,
                'amount'       => $amount,
                'note'         => $note,
                'status'       => 'succeeded',
                'idempotency_key' => $idempotencyKey,
            ]);

            // Ledger entries (debit sender, credit receiver)
            Transaction::create([
                'user_id'       => $from->id,
                'type'          => 'debit',
                'amount'        => $amount,
                'balance_after' => $from->balance,
                'related_type'  => Transfer::class,
                'related_id'    => $transfer->id,
                'meta'          => ['to_user_id' => $to->id, 'note' => $note],
            ]);

            Transaction::create([
                'user_id'       => $to->id,
                'type'          => 'credit',
                'amount'        => $amount,
                'balance_after' => $to->balance,
                'related_type'  => Transfer::class,
                'related_id'    => $transfer->id,
                'meta'          => ['from_user_id' => $from->id, 'note' => $note],
            ]);

            return $transfer;
        });
    }
}
