<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Support\Facades\Hash;

class User extends Authenticatable
{
    use HasFactory, Notifiable;

    protected $fillable = [
        'name',
        'email',
        'phone',
        'avatar',
        'password',
        'role',
        'balance',
        'reward_points',
        'login_pin',
        'device_token',
        'status',
        'device_lock',
    ];

    public function deviceLocked(): bool
{
    return (bool) $this->device_lock;
}


    protected $hidden = [
        'password',
        'remember_token',
        'login_pin',
        'device_token',
    ];

    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'balance'           => 'decimal:2',
            'reward_points'     => 'integer',
        ];
    }

    // ---- scopes / helpers ----

    public function scopeByPhone($q, string $phone)
    {
        return $q->where('phone', preg_replace('/\D/', '', $phone));
    }

    public function isAdmin(): bool
    {
        return $this->role === 'admin';
    }

    public function scopeAdmins($q)
    {
        return $q->where('role', 'admin');
    }

    public function isBlocked(): bool
    {
        return $this->status === 'blocked';
    }

    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    // ---- relations ----

    public function deposits()
    {
        return $this->hasMany(Deposit::class);
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class);
    }

    public function purchases()
    {
        return $this->hasMany(Purchase::class);
    }

    public function notices()
    {
        return $this->belongsToMany(\App\Models\Notice::class)
            ->withPivot('read_at')
            ->withTimestamps()
            ->orderByDesc('notices.created_at');
    }

    // ---- mutators ----

    // Password hash ফিক্স
    public function setPasswordAttribute($value): void
    {
        $value = (string) $value;

        $info = password_get_info($value);

        if (! empty($info['algo'])) {
            $this->attributes['password'] = Hash::needsRehash($value)
                ? Hash::make($value)
                : $value;

            return;
        }

        $this->attributes['password'] = Hash::make($value);
    }

    // PIN hash
    public function setLoginPinAttribute($value): void
    {
        if (empty($value)) {
            $this->attributes['login_pin'] = null;
            return;
        }

        $value = (string) $value;
        $info  = password_get_info($value);

        if (! empty($info['algo'])) {
            $this->attributes['login_pin'] = Hash::needsRehash($value)
                ? Hash::make($value)
                : $value;

            return;
        }

        $this->attributes['login_pin'] = Hash::make($value);
    }
}
