<?php

namespace App\Http\Controllers\backend;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;

class UserController extends Controller
{
    public function list(Request $request)
    {
        $q = User::query()
            ->when($request->filled('search'), function($query) use($request){
                $s = '%'.$request->string('search').'%';
                $query->where(function($q2) use($s){
                    $q2->where('name','like',$s)
                        ->orWhere('email','like',$s)
                        ->orWhere('phone','like',$s);
                });
            })
            ->orderBy('id','desc');

        $users = $q->paginate(12)->withQueryString();
        return view('admin.users.users_list', compact('users'));
    }

    public function create()
    {
        return view('admin.users.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'     => ['required','string','max:191'],
            'email'    => ['nullable','email','max:191','unique:users,email'],
            'phone'    => ['nullable','string','max:30','unique:users,phone'],
            'role'     => ['required', Rule::in(['user','admin'])],
            'status'   => ['required', Rule::in(['active','blocked'])],
            'password' => ['required','min:6'],
            'avatar'   => ['nullable','image','max:2048'],
        ]);

        if ($request->hasFile('avatar')) {
            $data['avatar'] = $request->file('avatar')->store('avatars','public');
        }
        $data['password'] = Hash::make($data['password']);

        User::create($data);
        return redirect()->route('admin.users.list')->with('success','User created.');
    }

    public function edit(User $user)
    {
        return view('admin.users.edit', compact('user'));
    }

    public function update(Request $request, User $user)
    {
        $data = $request->validate([
            'name'     => ['required','string','max:191'],
            'email'    => ['nullable','email','max:191', Rule::unique('users','email')->ignore($user->id)],
            'phone'    => ['nullable','string','max:30', Rule::unique('users','phone')->ignore($user->id)],
            'role'     => ['required', Rule::in(['user','admin'])],
            'status'   => ['required', Rule::in(['active','blocked'])],
            'password' => ['nullable','min:6'],
            'avatar'   => ['nullable','image','max:2048'],
        ]);

        if ($request->filled('password')) {
            $data['password'] = Hash::make($data['password']);
        } else {
            unset($data['password']);
        }

        if ($request->hasFile('avatar')) {
            if ($user->avatar) {
                Storage::disk('public')->delete($user->avatar);
            }
            $data['avatar'] = $request->file('avatar')->store('avatars','public');
        }

        $user->update($data);
        return redirect()->route('admin.users.list')->with('success','User updated.');
    }

    public function destroy(User $user)
    {
        if ($user->avatar) {
            Storage::disk('public')->delete($user->avatar);
        }
        $user->delete();
        return redirect()->route('admin.users.list')->with('success','User deleted.');
    }

    /**
     * ✅ FIX: আর toggle না, frontend থেকে যে status আসবে, সেটাই set করব।
     * তাই auto flip বা reload এ নিজে থেকে বদলাবে না।
     */
    public function toggleStatus(User $user, Request $request)
    {
        $data = $request->validate([
            'status' => ['required', Rule::in(['active','blocked'])],
        ]);

        $newStatus = $data['status'];

        // শুধু পরিবর্তন হলে save করব
        if ($user->status !== $newStatus) {
            $user->status = $newStatus;
            $user->save();
        }

        if ($request->expectsJson()) {
            return response()->json([
                'ok'     => true,
                'id'     => $user->id,
                'status' => $user->status,
                'label'  => ucfirst($user->status),
            ]);
        }

        return back()->with('success', "Status: {$user->status}");
    }

    public function deviceLock(Request $request, User $user)
    {
        // ✅ admin এর জন্য device lock apply হবে না
        if ($user->role === 'admin') {
            return response()->json([
                'success' => false,
                'message' => 'Admin একাউন্টের জন্য ডিভাইস লক প্রযোজ্য নয়।',
            ], 422);
        }

        // checkbox checked হলে = allow new device ON (অর্থাৎ device_lock = false)
        $allowNew = $request->boolean('allow_new_device');

        // allowNew = true  => device_lock = false  (unlock)
        // allowNew = false => device_lock = true   (lock)
        $user->device_lock = $allowNew ? false : true;
        $user->save();

        return response()->json([
            'success'      => true,
            'device_lock'  => (bool) $user->device_lock,
            'allow_new'    => $allowNew,
            'message'      => $user->device_lock
                ? 'ডিভাইস আবার লক করা হয়েছে (শুধু বর্তমান ডিভাইসে লগইন করা যাবে)।'
                : 'নতুন ডিভাইস থেকে একবার লগইনের অনুমতি দেওয়া হয়েছে।',
        ]);
    }
}
