<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Transfer;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class TransferManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-transfers');
    }

    /**
     * GET /admin/transfers
     * Filters: status=[pending|succeeded|failed|reversed], q=search (note/idempotency/from/to)
     */
    public function index(Request $request)
    {
        $request->validate([
            'status' => ['nullable','in:pending,succeeded,failed,reversed'],
            'q'      => ['nullable','string','max:100'],
        ]);

        $query = Transfer::with(['fromUser','toUser'])->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function ($qq) use ($q) {
                $qq->where('note', 'like', "%{$q}%")
                   ->orWhere('idempotency_key', 'like', "%{$q}%")
                   ->orWhereHas('fromUser', fn($fq) => $fq->where('name', 'like', "%{$q}%"))
                   ->orWhereHas('toUser',   fn($tq) => $tq->where('name', 'like', "%{$q}%"));
            });
        }

        $transfers = $query->paginate(30)->withQueryString();

        return view('admin.transfers.index', compact('transfers'));
    }

    /**
     * POST /admin/transfers/{transfer}/approve
     * Body: note (nullable string)
     * Only allowed when status = pending.
     * Side effects: debit sender, credit recipient (atomic), mark as succeeded, log transactions.
     */
    public function approve(Transfer $transfer, Request $request)
    {
        $data = $request->validate([
            'note' => ['nullable', 'string', 'max:500'],
        ]);

        $adminNote = $data['note'] ?? null;

        DB::transaction(function () use ($transfer, $adminNote) {
            /** @var \App\Models\Transfer $tx */
            $tx = Transfer::whereKey($transfer->id)->lockForUpdate()->first();

            if (!$tx || $tx->status !== 'pending') {
                abort(400, 'Only pending transfers can be approved.');
            }

            // Lock users in stable order to prevent deadlocks
            $ids = [$tx->from_user_id, $tx->to_user_id];
            sort($ids);

            $u1 = DB::table('users')->where('id', $ids[0])->lockForUpdate()->first();
            $u2 = DB::table('users')->where('id', $ids[1])->lockForUpdate()->first();

            // Re-fetch Eloquent models for update convenience
            $from = $tx->fromUser()->lockForUpdate()->first(); // original sender
            $to   = $tx->toUser()->lockForUpdate()->first();   // original recipient

            // Ensure sender has sufficient balance
            if (($from->balance ?? 0) < $tx->amount) {
                abort(422, 'Sender balance is insufficient to approve this transfer.');
            }

            // 1) Mark succeeded (with admin note appended)
            $note = $tx->note ? $tx->note . ' | ' : '';
            $note .= 'Approved';
            if (!empty($adminNote)) {
                $note .= ': ' . $adminNote;
            }

            $tx->update([
                'status' => 'succeeded',
                'note'   => trim($note),
            ]);

            // 2) Move funds: sender -> recipient
            $from->balance = $from->balance - $tx->amount;
            $to->balance   = $to->balance + $tx->amount;
            $from->save();
            $to->save();

            // 3) Transaction logs
            // Debit sender
            Transaction::create([
                'user_id'       => $from->id,
                'type'          => 'debit',
                'amount'        => $tx->amount,
                'balance_after' => $from->balance,
                'related_type'  => Transfer::class,
                'related_id'    => $tx->id,
                'meta'          => ['approved' => true, 'direction' => 'from->to'],
            ]);

            // Credit recipient
            Transaction::create([
                'user_id'       => $to->id,
                'type'          => 'credit',
                'amount'        => $tx->amount,
                'balance_after' => $to->balance,
                'related_type'  => Transfer::class,
                'related_id'    => $tx->id,
                'meta'          => ['approved' => true, 'direction' => 'from->to'],
            ]);
        });

        return back()->with('ok', 'Transfer approved successfully.');
    }

    /**
     * POST /admin/transfers/{transfer}/reject
     * Body: note (required string)
     * Only allowed when status = pending.
     * Side effects: mark as failed, no balance changes.
     */
    public function reject(Transfer $transfer, Request $request)
    {
        $data = $request->validate([
            'note' => ['required', 'string', 'max:500'],
        ]);

        DB::transaction(function () use ($transfer, $data) {
            /** @var \App\Models\Transfer $tx */
            $tx = Transfer::whereKey($transfer->id)->lockForUpdate()->first();

            if (!$tx || $tx->status !== 'pending') {
                abort(400, 'Only pending transfers can be rejected.');
            }

            $note = $tx->note ? $tx->note . ' | ' : '';
            $note .= 'Rejected: ' . $data['note'];

            // If you prefer a dedicated "rejected" status, change 'failed' -> 'rejected'
            $tx->update([
                'status' => 'failed',
                'note'   => trim($note),
            ]);
        });

        return back()->with('ok', 'Transfer rejected successfully.');
    }

    /**
     * POST /admin/transfers/{transfer}/reverse
     * Body: note (required string)
     * Only allowed when status = succeeded.
     * Side effects: debit recipient, credit sender (atomic), mark as reversed, log transactions.
     */
    public function reverse(Transfer $transfer, Request $request)
    {
        $data = $request->validate([
            'note' => ['required','string','max:500'],
        ]);

        DB::transaction(function () use ($transfer, $data) {
            // Lock the transfer row
            /** @var \App\Models\Transfer $tx */
            $tx = Transfer::whereKey($transfer->id)->lockForUpdate()->first();

            if (!$tx || $tx->status !== 'succeeded') {
                abort(400, 'Only succeeded transfers can be reversed.');
            }

            // Lock users in stable order to prevent deadlocks
            $ids = [$tx->from_user_id, $tx->to_user_id];
            sort($ids);

            $u1 = DB::table('users')->where('id',$ids[0])->lockForUpdate()->first();
            $u2 = DB::table('users')->where('id',$ids[1])->lockForUpdate()->first();

            // Re-fetch Eloquent models for update convenience
            $from = $tx->fromUser()->lockForUpdate()->first(); // original sender
            $to   = $tx->toUser()->lockForUpdate()->first();   // original recipient

            // Ensure recipient has sufficient balance to reverse (optional guard)
            if (($to->balance ?? 0) < $tx->amount) {
                abort(422, 'Recipient balance is insufficient to reverse this transfer.');
            }

            // 1) Mark reversed (with admin note appended)
            $tx->update([
                'status' => 'reversed',
                'note'   => trim(($tx->note ? ($tx->note . ' | ') : '') . 'Reversed: ' . $data['note']),
            ]);

            // 2) Move funds back: recipient -> sender
            $to->balance   = $to->balance - $tx->amount;
            $from->balance = $from->balance + $tx->amount;
            $to->save();
            $from->save();

            // 3) Transaction logs
            // Debit recipient
            Transaction::create([
                'user_id'       => $to->id,
                'type'          => 'debit',
                'amount'        => $tx->amount,
                'balance_after' => $to->balance,
                'related_type'  => Transfer::class,
                'related_id'    => $tx->id,
                'meta'          => ['reversal' => true, 'direction' => 'to->from'],
            ]);

            // Credit sender
            Transaction::create([
                'user_id'       => $from->id,
                'type'          => 'credit',
                'amount'        => $tx->amount,
                'balance_after' => $from->balance,
                'related_type'  => Transfer::class,
                'related_id'    => $tx->id,
                'meta'          => ['reversal' => true, 'direction' => 'to->from'],
            ]);
        });

        return back()->with('ok', 'Transfer reversed successfully.');
    }
}
