<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Recharge;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RechargeManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-recharges');
    }

    /**
     * GET /admin/recharges
     * Filters: status=[pending|succeeded|failed], service_type=[prepaid|postpaid], q in msisdn/reference/txn/user
     */
    public function index(Request $request)
    {
        $request->validate([
            'status'       => ['nullable','in:pending,succeeded,failed'],
            'service_type' => ['nullable','in:prepaid,postpaid'],
            'q'            => ['nullable','string','max:100'],
        ]);

        $query = Recharge::with('user')->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($request->filled('service_type')) {
            $query->where('service_type', $request->string('service_type')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function($qq) use ($q) {
                $qq->where('msisdn','like',"%$q%")
                   ->orWhere('reference','like',"%$q%")
                   ->orWhere('txn_id','like',"%$q%")
                   ->orWhereHas('user', fn($uq) => $uq->where('name','like',"%$q%"));
            });
        }

        $recharges = $query->paginate(30)->withQueryString();

        return view('admin.recharges.index', compact('recharges'));
    }

    /**
     * POST /admin/recharges/{recharge}/succeed
     * Body: txn_id (nullable), reference (nullable)
     * Only allowed from pending.
     */
    public function succeed(Recharge $recharge, Request $request)
    {
        $data = $request->validate([
            'txn_id'    => ['nullable','string','max:100'],
            'reference' => ['nullable','string','max:100'],
        ]);

        DB::transaction(function () use ($recharge, $data) {
            $r = Recharge::whereKey($recharge->id)->lockForUpdate()->first();
            if (!$r || $r->status !== 'pending') {
                abort(400, 'Only pending recharges can be marked succeeded.');
            }

            // (Optional) Deduct from user's wallet if required:
            // $user = $r->user()->lockForUpdate()->first();
            // if (($user->balance ?? 0) < $r->amount) { abort(422,'Insufficient balance'); }
            // $user->balance -= $r->amount; $user->save();
            // (Optional) add Transaction log...

            $r->update([
                'status'    => 'succeeded',
                'txn_id'    => $data['txn_id'] ?? $r->txn_id,
                'reference' => $data['reference'] ?? $r->reference,
            ]);
        });

        return back()->with('ok', 'Recharge marked as succeeded.');
    }

    /**
     * POST /admin/recharges/{recharge}/fail
     * Body: reference (nullable) — failure note/reference
     * Only allowed from pending.
     */
    public function fail(Recharge $recharge, Request $request)
    {
        $data = $request->validate([
            'reference' => ['nullable','string','max:100'],
        ]);

        DB::transaction(function () use ($recharge, $data) {
            $r = Recharge::whereKey($recharge->id)->lockForUpdate()->first();
            if (!$r || $r->status !== 'pending') {
                abort(400, 'Only pending recharges can be marked failed.');
            }

            $r->update([
                'status'    => 'failed',
                'reference' => $data['reference'] ?? $r->reference,
            ]);
        });

        return back()->with('ok', 'Recharge marked as failed.');
    }
}
