<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\Purchase;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class PurchaseManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-purchases');
    }

    /**
     * GET /admin/purchases
     * Filters: status=[pending|succeeded|failed], q in msisdn/reference/user/offer
     */
    public function index(Request $request)
    {
        $request->validate([
            'status' => ['nullable','in:pending,succeeded,failed'],
            'q'      => ['nullable','string','max:100'],
        ]);

        $query = Purchase::with(['user','offer'])->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function($qq) use ($q) {
                $qq->where('msisdn','like',"%$q%")
                   ->orWhere('reference','like',"%$q%")
                   ->orWhere('txn_id','like',"%$q%")
                   ->orWhereHas('user', fn($uq) => $uq->where('name','like',"%$q%"))
                   ->orWhereHas('offer', fn($oq) => $oq->where('title','like',"%$q%"));
            });
        }

        $purchases = $query->paginate(30)->withQueryString();

        return view('admin.purchases.index', compact('purchases'));
    }

    /**
     * POST /admin/purchases/{purchase}/succeed
     * Body: txn_id (nullable string), reference (nullable string)
     * Only allowed from pending.
     */
    public function succeed(Purchase $purchase, Request $request)
    {
        $data = $request->validate([
            'txn_id'    => ['nullable','string','max:100'],
            'reference' => ['nullable','string','max:100'],
        ]);

        DB::transaction(function () use ($purchase, $data) {
            $p = Purchase::whereKey($purchase->id)->lockForUpdate()->first();
            if (!$p || $p->status !== 'pending') {
                abort(400, 'Only pending purchases can be marked succeeded.');
            }

            // If you need to DEBIT user's wallet here, this is the place.
            // $user = $p->user()->lockForUpdate()->first();
            // if (($user->balance ?? 0) < $p->price) { abort(422,'Insufficient balance'); }
            // $user->balance -= $p->price; $user->save();
            // (Optional) create Transaction log...

            $p->update([
                'status'    => 'succeeded',
                'txn_id'    => $data['txn_id'] ?? $p->txn_id,
                'reference' => $data['reference'] ?? $p->reference,
            ]);
        });

        return back()->with('ok', 'Marked as succeeded.');
    }

    /**
     * POST /admin/purchases/{purchase}/fail
     * Body: reference (nullable string) — e.g. failure reason code/ref
     * Only allowed from pending.
     */
    public function fail(Purchase $purchase, Request $request)
    {
        $data = $request->validate([
            'reference' => ['nullable','string','max:100'],
        ]);

        DB::transaction(function () use ($purchase, $data) {
            $p = Purchase::whereKey($purchase->id)->lockForUpdate()->first();
            if (!$p || $p->status !== 'pending') {
                abort(400, 'Only pending purchases can be marked failed.');
            }

            $p->update([
                'status'    => 'failed',
                'reference' => $data['reference'] ?? $p->reference,
            ]);
        });

        return back()->with('ok', 'Marked as failed.');
    }
}
