<?php


namespace App\Http\Controllers\backend;

use App\Http\Controllers\Controller;

use App\Models\Offer;
use App\Models\Operator;
use Illuminate\Http\Request;

class OfferController extends Controller
{
    public function index(Request $request)
    {
        $q = Offer::with('operator')->orderBy('sort_order')->orderByDesc('id');

        if ($s = $request->get('q')) {
            $q->where('title', 'like', "%{$s}%");
        }
        if ($op = $request->get('operator_id')) {
            $q->where('operator_id', $op);
        }
        if ($st = $request->get('status')) {
            $q->where('status', $st);
        }

        $offers    = $q->paginate(12)->withQueryString();
        $operators = Operator::orderBy('name')->get(['id','name']);
        return view('backend.offers.index', compact('offers','operators'));
    }

    public function create()
    {
        $operators = Operator::orderBy('name')->get(['id','name']);
        return view('backend.offers.create', compact('operators'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'operator_id'   => 'required|exists:operators,id',
            'title'         => 'required|string|max:255',
            'regular_price' => 'required|numeric|min:0',
            'sale_price'    => 'required|numeric|min:0|lte:regular_price',
            'data_gb'       => 'nullable|integer|min:0',
            'minutes'       => 'nullable|integer|min:0',
            'validity_days' => 'nullable|integer|min:0',
            'status'        => 'required|in:active,inactive',
            'sort_order'    => 'nullable|integer|min:0',
        ]);

        $data['sort_order'] = $data['sort_order'] ?? 0;

        Offer::create($data);
        return redirect(url('/admin/offers'))->with('success', 'Offer created.');
    }

    public function show($id)
    {
        $offer = Offer::with('operator')->findOrFail($id);
        return view('backend.offers.show', compact('offer'));
    }

    public function edit($id)
    {
        $offer     = Offer::findOrFail($id);
        $operators = Operator::orderBy('name')->get(['id','name']);
        return view('backend.offers.edit', compact('offer','operators'));
    }

    public function update(Request $request, $id)
    {
        $offer = Offer::findOrFail($id);

        $data = $request->validate([
            'operator_id'   => 'required|exists:operators,id',
            'title'         => 'required|string|max:255',
            'regular_price' => 'required|numeric|min:0',
            'sale_price'    => 'required|numeric|min:0|lte:regular_price',
            'data_gb'       => 'nullable|integer|min:0',
            'minutes'       => 'nullable|integer|min:0',
            'validity_days' => 'nullable|integer|min:0',
            'status'        => 'required|in:active,inactive',
            'sort_order'    => 'nullable|integer|min:0',
        ]);

        $data['sort_order'] = $data['sort_order'] ?? 0;

        $offer->update($data);
        return redirect(url('/admin/offers'))->with('success', 'Offer updated.');
    }

    public function destroy($id)
    {
        Offer::findOrFail($id)->delete();
        return redirect(url('/admin/offers'))->with('success', 'Offer deleted.');
    }

    public function toggle($id)
    {
        $offer = Offer::findOrFail($id);
        $offer->update(['status' => $offer->status === 'active' ? 'inactive' : 'active']);
        return back()->with('success', 'Status toggled.');
    }
}
