<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\MobileTransaction;
use App\Models\User;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MobileTransactionManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-mobile-transactions');
    }

    public function index(Request $request)
    {
        $request->validate([
            'status'  => ['nullable','in:pending,approved,rejected'],
            'channel' => ['nullable','in:agent,personal'],
            'q'       => ['nullable','string','max:100'],
        ]);

        $query = MobileTransaction::with(['user','gateway'])->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }
        if ($request->filled('channel')) {
            $query->where('channel', $request->string('channel')->toString());
        }
        if ($q = $request->string('q')->toString()) {
            $query->where(function($qq) use ($q) {
                $qq->where('account_number','like',"%$q%")
                   ->orWhere('reference','like',"%$q%")
                   ->orWhereHas('user', fn($uq) => $uq->where('name','like',"%$q%"))
                   ->orWhereHas('gateway', fn($gq) => $gq->where('name','like',"%$q%"));
            });
        }

        $transactions = $query->paginate(30)->withQueryString();

        return view('admin.mobile_tx.index', compact('transactions'));
    }

    /**
     * POST /admin/mobile-transactions/{tx}/approve
     *
     * 👉 এখন থেকে:
     *  - submit সময়ে wallet থেকে টাকা কাটা হবে
     *  - approve সময়ে আর balance change হবে না
     *  - শুধু status = approved, reward points, audit Transaction (amount = 0)
     */
    public function approve(MobileTransaction $tx, Request $request)
    {
        abort_if($tx->status !== 'pending', 400, 'Already processed.');

        $data = $request->validate([
            'commission_amount' => ['nullable','numeric','min:0'],
            'admin_note'        => ['nullable','string','max:1000'],
        ]);

        $awarded = 0;

        try {
            DB::transaction(function () use ($tx, $data, &$awarded) {

                /** @var MobileTransaction $lockedTx */
                $lockedTx = MobileTransaction::whereKey($tx->id)->lockForUpdate()->firstOrFail();

                if ($lockedTx->status !== 'pending') {
                    throw new \RuntimeException('ALREADY_PROCESSED');
                }

                /** @var User $user */
                $user = User::whereKey($lockedTx->user_id)->lockForUpdate()->firstOrFail();

                $amount     = (float) $lockedTx->amount;
                $commission = $data['commission_amount'] ?? $lockedTx->commission_amount;

                // ❌ আগে এখানে balance check + debit ছিল
                // ✅ এখন approve এ কোনো balance change হবে না

                // ✅ Approve + commission + admin_note update
                $lockedTx->update([
                    'status'            => 'approved',
                    'commission_amount' => $commission,
                    'admin_note'        => $data['admin_note'] ?? $lockedTx->admin_note,
                ]);

                // ===== Reward Points (mobile_banking_comm %) =====
                $settings = DB::table('reward_settings')
                    ->where('is_active', 1)
                    ->orderByDesc('id')
                    ->first();

                $percent = $settings ? (float) $settings->mobile_banking_comm : 0.0;

                $awarded = 0;

                if ($percent > 0 && $amount > 0) {
                    $awarded = (int) floor(($amount * $percent) / 100);

                    if ($awarded > 0) {
                        DB::table('users')
                            ->where('id', $lockedTx->user_id)
                            ->update([
                                'reward_points' => DB::raw('reward_points + '.$awarded),
                            ]);

                        if ($this->tableHasColumn('mobile_transactions', 'reward_points_awarded')) {
                            DB::table('mobile_transactions')
                                ->where('id', $lockedTx->id)
                                ->update(['reward_points_awarded' => $awarded]);
                        }
                    }
                }

                // ✅ Approve লগের জন্য Transaction row
                // amount = 0, কারণ আসল debit submit সময় হয়ে গেছে
                Transaction::create([
                    'user_id'       => $user->id,
                    'type'          => 'debit',
                    'amount'        => 0,
                    'balance_after' => $user->balance, // কোনো পরিবর্তন নেই
                    'related_type'  => MobileTransaction::class,
                    'related_id'    => $lockedTx->id,
                    'meta'          => json_encode([
                        'gateway_id'            => $lockedTx->gateway_id,
                        'gateway_code'          => optional($lockedTx->gateway)->code,
                        'gateway_name'          => optional($lockedTx->gateway)->name,
                        'channel'               => $lockedTx->channel,
                        'account_number'        => $lockedTx->account_number,
                        'reference'             => $lockedTx->reference,
                        'commission'            => (float) $commission,
                        'status'                => 'approved',
                        'admin_note'            => $lockedTx->admin_note,
                        'reward_points_awarded' => $awarded,
                        'note'                  => 'Mobile transaction approved (amount was debited at request time).',
                    ]),
                ]);
            });
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'ALREADY_PROCESSED') {
                return back()->with('warning', 'Mobile transaction already processed.');
            }
            throw $e;
        }

        return back()->with('ok', 'Mobile transaction approved.');
    }

    /**
     * POST /admin/mobile-transactions/{tx}/reject
     *
     * 👉 এখন থেকে:
     *  - submit এ যে টাকা কাটা হয়েছিল, reject এ সেই টাকা wallet এ ফেরত যাবে
     *  - status = rejected
     *  - Transaction row তৈরি হবে (type=credit, amount=original amount)
     */
    public function reject(MobileTransaction $tx, Request $request)
    {
        abort_if($tx->status !== 'pending', 400, 'Already processed.');

        $data = $request->validate([
            'admin_note' => ['required','string','max:1000'],
        ]);

        try {
            DB::transaction(function () use ($tx, $data) {
                /** @var MobileTransaction $lockedTx */
                $lockedTx = MobileTransaction::whereKey($tx->id)->lockForUpdate()->firstOrFail();

                if ($lockedTx->status !== 'pending') {
                    throw new \RuntimeException('ALREADY_PROCESSED');
                }

                /** @var User $user */
                $user = User::whereKey($lockedTx->user_id)->lockForUpdate()->firstOrFail();

                $amount = (float) $lockedTx->amount;

                // ✅ Wallet refund (submit এ যেটা কেটেছিল, সেটা ফেরত)
                $newBalance = bcadd($user->balance, $amount, 2);
                $user->update([
                    'balance' => $newBalance,
                ]);

                // status + note আপডেট
                $lockedTx->update([
                    'status'     => 'rejected',
                    'admin_note' => $data['admin_note'],
                ]);

                // ✅ REJECT সময় Transaction row (refund credit)
                Transaction::create([
                    'user_id'       => $user->id,
                    'type'          => 'credit',
                    'amount'        => $amount,
                    'balance_after' => $newBalance,
                    'related_type'  => MobileTransaction::class,
                    'related_id'    => $lockedTx->id,
                    'meta'          => json_encode([
                        'gateway_id'      => $lockedTx->gateway_id,
                        'gateway_code'    => optional($lockedTx->gateway)->code,
                        'gateway_name'    => optional($lockedTx->gateway)->name,
                        'channel'         => $lockedTx->channel,
                        'account_number'  => $lockedTx->account_number,
                        'reference'       => $lockedTx->reference,
                        'status'          => 'rejected',
                        'admin_note'      => $data['admin_note'],
                        'note'            => 'Mobile transaction request rejected and refunded to wallet.',
                    ]),
                ]);
            });
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'ALREADY_PROCESSED') {
                return back()->with('warning', 'Mobile transaction already processed.');
            }
            throw $e;
        }

        return back()->with('ok', 'Mobile transaction rejected.');
    }

    protected function tableHasColumn(string $table, string $column): bool
    {
        try {
            /** @var \Illuminate\Database\Schema\Builder $schema */
            $schema = app('db.schema');
            return $schema->hasColumn($table, $column);
        } catch (\Throwable $e) {
            return false;
        }
    }
}
