<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\LoanApplication;
use App\Models\LoanRepayment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class LoanManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-loans');
    }

    /**
     * GET /admin/loans
     * Filters: status=[pending|approved|rejected], q in user name/id
     */
    public function index(Request $request)
    {
        $request->validate([
            'status' => ['nullable','in:pending,approved,rejected'],
            'q'      => ['nullable','string','max:100'],
        ]);

        $query = LoanApplication::with('user')->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function($qq) use ($q) {
                $qq->where('id', intval($q) ?: 0)
                   ->orWhereHas('user', fn($uq) => $uq->where('name','like',"%$q%"));
            });
        }

        $loans = $query->paginate(30)->withQueryString();
        return view('admin.loans.index', compact('loans'));
    }

    /**
     * POST /admin/loans/{loan}/approve
     * Generates repayment schedule if none exists
     */
    public function approve(LoanApplication $loan, Request $request)
    {
        abort_if($loan->status !== 'pending', 400, 'Already processed.');

        DB::transaction(function () use ($loan) {
            // Approve application
            $loan->update(['status' => 'approved']);

            // Create schedule only if not exists
            if ($loan->repayments()->count() === 0) {
                $period = max(1, (int)$loan->loan_period);
                $total  = (int)$loan->estimated_total;

                // Simple equal installment split (2 decimals)
                $per   = round($total / $period, 2);
                $sum   = 0;
                $rows  = [];
                $start = Carbon::now()->addMonth(); // first due next month

                for ($i=1; $i <= $period; $i++) {
                    $amt = ($i < $period) ? $per : round($total - $sum, 2); // last row adjust
                    $sum += $amt;

                    $rows[] = [
                        'due_date'             => $start->copy()->addMonths($i-1)->toDateString(),
                        'amount'               => $amt,
                        'status'               => 'due',
                        'created_at'           => now(),
                        'updated_at'           => now(),
                    ];
                }

                foreach ($rows as $r) {
                    $loan->repayments()->create($r);
                }
            }
        });

        return back()->with('ok', 'Loan approved & schedule generated.');
    }

    /**
     * POST /admin/loans/{loan}/reject
     */
    public function reject(LoanApplication $loan)
    {
        abort_if($loan->status !== 'pending', 400, 'Already processed.');
        $loan->update(['status' => 'rejected']);
        return back()->with('ok', 'Loan rejected.');
    }

    /**
     * GET /admin/loans/{loan}/repayments
     */
    public function repayments(LoanApplication $loan)
    {
        $loan->load('user');
        $repayments = $loan->repayments()->orderBy('due_date')->paginate(50)->withQueryString();
        return view('admin.loans.repayments', compact('loan', 'repayments'));
    }

    /**
     * POST /admin/loans/repayments/{repayment}/mark-paid
     */
    public function markPaid(LoanRepayment $repayment)
    {
        DB::transaction(function () use ($repayment) {
            $rp = LoanRepayment::whereKey($repayment->id)->lockForUpdate()->first();
            if (!$rp || $rp->status !== 'due') {
                abort(400, 'Only due installments can be marked paid.');
            }
            $rp->update([
                'status'  => 'paid',
                'paid_at' => now(),
            ]);
        });

        return back()->with('ok', 'Installment marked as paid.');
    }
}
