<?php

namespace App\Http\Controllers\backend;

use App\Http\Controllers\Controller;

use App\Models\ExchangeCountry;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class ExchangeCountryController extends Controller
{
    public function index(Request $request)
    {
        $q = ExchangeCountry::query();

        if ($s = $request->get('q')) {
            $q->where(function($w) use ($s){
                $w->where('name','like',"%{$s}%")
                  ->orWhere('slug','like',"%{$s}%")
                  ->orWhere('currency_name','like',"%{$s}%");
            });
        }
        if ($request->filled('active')) {
            $q->where('is_active', (bool)$request->get('active'));
        }

        $countries = $q->orderBy('position')->orderBy('name')
                       ->paginate(12)->withQueryString();

        return view('backend.exchange_countries.index', compact('countries'));
    }

    public function create()
    {
        return view('backend.exchange_countries.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name'          => ['required','string','max:255'],
            'slug'          => ['nullable','string','max:191','unique:exchange_countries,slug'],
            'currency_name' => ['required','string','max:191'],
            'unit_label'    => ['nullable','string','max:50'],
            'rate_bdt'      => ['required','numeric','min:0'],
            'flag_path'     => ['nullable','string','max:600'], // url/emoji/text
            'flag_file'     => ['nullable','image','mimes:jpg,jpeg,png,webp,svg','max:2048'],
            'header_bg'     => ['nullable','string','max:50'],  // e.g. #4C0519 or gradient class
            'share_title_bn'=> ['nullable','string','max:255'],
            'share_title_en'=> ['nullable','string','max:255'],
            'share_desc_bn' => ['nullable','string'],
            'share_desc_en' => ['nullable','string'],
            'is_active'     => ['required','in:0,1'],
            'position'      => ['nullable','integer','min:0'],
        ]);

        if (empty($data['slug'])) $data['slug'] = Str::slug($data['name']);
        if (empty($data['unit_label'])) $data['unit_label'] = '1';
        if (!isset($data['position'])) $data['position'] = 0;

        if ($request->hasFile('flag_file')) {
            $data['flag_path'] = $request->file('flag_file')->store('uploads/flags','public');
        }

        ExchangeCountry::create($data);
        return redirect(url('/admin/exchange-countries'))->with('success','Country added.');
    }

    public function show($id)
    {
        $row = ExchangeCountry::findOrFail($id);
        return view('backend.exchange_countries.show', compact('row'));
    }

    public function edit($id)
    {
        $row = ExchangeCountry::findOrFail($id);
        return view('backend.exchange_countries.edit', compact('row'));
    }

    public function update(Request $request, $id)
    {
        $row = ExchangeCountry::findOrFail($id);

        $data = $request->validate([
            'name'          => ['required','string','max:255'],
            'slug'          => ['nullable','string','max:191', Rule::unique('exchange_countries','slug')->ignore($row->id)],
            'currency_name' => ['required','string','max:191'],
            'unit_label'    => ['nullable','string','max:50'],
            'rate_bdt'      => ['required','numeric','min:0'],
            'flag_path'     => ['nullable','string','max:600'],
            'flag_file'     => ['nullable','image','mimes:jpg,jpeg,png,webp,svg','max:2048'],
            'remove_flag'   => ['nullable','boolean'],
            'header_bg'     => ['nullable','string','max:50'],
            'share_title_bn'=> ['nullable','string','max:255'],
            'share_title_en'=> ['nullable','string','max:255'],
            'share_desc_bn' => ['nullable','string'],
            'share_desc_en' => ['nullable','string'],
            'is_active'     => ['required','in:0,1'],
            'position'      => ['nullable','integer','min:0'],
        ]);

        if (empty($data['slug'])) $data['slug'] = Str::slug($data['name']);
        if (empty($data['unit_label'])) $data['unit_label'] = '1';
        if (!isset($data['position'])) $data['position'] = 0;

        if ($request->boolean('remove_flag') && $row->flag_path && Storage::disk('public')->exists($row->flag_path)) {
            Storage::disk('public')->delete($row->flag_path);
            $data['flag_path'] = null;
        }

        if ($request->hasFile('flag_file')) {
            $path = $request->file('flag_file')->store('uploads/flags','public');
            if ($row->flag_path && Storage::disk('public')->exists($row->flag_path)) {
                Storage::disk('public')->delete($row->flag_path);
            }
            $data['flag_path'] = $path;
        }

        $row->update($data);
        return redirect(url('/admin/exchange-countries'))->with('success','Country updated.');
    }

    public function destroy($id)
    {
        $row = ExchangeCountry::findOrFail($id);
        if ($row->flag_path && Storage::disk('public')->exists($row->flag_path)) {
            Storage::disk('public')->delete($row->flag_path);
        }
        $row->delete();
        return redirect(url('/admin/exchange-countries'))->with('success','Country deleted.');
    }

    public function toggle($id)
    {
        $row = ExchangeCountry::findOrFail($id);
        $row->update(['is_active' => !$row->is_active]);
        return back()->with('success','Status toggled.');
    }
}
