<?php

namespace App\Http\Controllers\backend;

use App\Http\Controllers\Controller;

use App\Models\Biller;
use App\Models\BillerCategory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class BillerController extends Controller
{
    public function index(Request $request)
    {
        $q = Biller::with('category');

        if ($s = $request->get('q')) {
            $q->where(function($w) use ($s){
                $w->where('name','like',"%{$s}%")
                  ->orWhere('code','like',"%{$s}%")
                  ->orWhere('provider','like',"%{$s}%");
            });
        }
        if ($cid = $request->get('category_id')) {
            $q->where('biller_category_id', $cid);
        }
        if ($st = $request->get('status')) {
            $q->where('status', $st);
        }
        if ($tp = $request->get('type')) {
            $q->where('type', $tp);
        }

        $billers = $q->orderBy('sort_order')->orderBy('name')->paginate(12)->withQueryString();
        $categories = BillerCategory::orderBy('sort_order')->orderBy('name')->get(['id','name']);
        return view('backend.billers.index', compact('billers','categories'));
    }

    public function create()
    {
        $categories = BillerCategory::orderBy('sort_order')->orderBy('name')->get(['id','name']);
        return view('backend.billers.create', compact('categories'));
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'biller_category_id' => ['required','exists:biller_categories,id'],
            'name'       => ['required','string','max:255'],
            'code'       => ['required','string','max:191','unique:billers,code'],
            'provider'   => ['nullable','string','max:191'],
            'type'       => ['nullable', Rule::in(['prepaid','postpaid'])],
            'logo'       => ['nullable','string','max:500'], // url/emoji/text
            'logo_file'  => ['nullable','image','mimes:jpg,jpeg,png,webp,svg','max:2048'],
            'min_amount' => ['required','numeric','min:0'],
            'max_amount' => ['required','numeric','gte:min_amount'],
            'status'     => ['required', Rule::in(['active','inactive'])],
            'sort_order' => ['nullable','integer','min:0'],
        ]);

        $data['sort_order'] = $data['sort_order'] ?? 0;

        if ($request->hasFile('logo_file')) {
            $data['logo'] = $request->file('logo_file')->store('uploads/billers', 'public');
        }

        Biller::create($data);
        return redirect(url('/admin/billers'))->with('success', 'Biller created.');
    }

    public function show($id)
    {
        $biller = Biller::with('category')->findOrFail($id);
        return view('backend.billers.show', compact('biller'));
    }

    public function edit($id)
    {
        $biller = Biller::findOrFail($id);
        $categories = BillerCategory::orderBy('sort_order')->orderBy('name')->get(['id','name']);
        return view('backend.billers.edit', compact('biller','categories'));
    }

    public function update(Request $request, $id)
    {
        $biller = Biller::findOrFail($id);

        $data = $request->validate([
            'biller_category_id' => ['required','exists:biller_categories,id'],
            'name'       => ['required','string','max:255'],
            'code'       => ['required','string','max:191', Rule::unique('billers','code')->ignore($biller->id)],
            'provider'   => ['nullable','string','max:191'],
            'type'       => ['nullable', Rule::in(['prepaid','postpaid'])],
            'logo'       => ['nullable','string','max:500'],
            'logo_file'  => ['nullable','image','mimes:jpg,jpeg,png,webp,svg','max:2048'],
            'remove_logo'=> ['nullable','boolean'],
            'min_amount' => ['required','numeric','min:0'],
            'max_amount' => ['required','numeric','gte:min_amount'],
            'status'     => ['required', Rule::in(['active','inactive'])],
            'sort_order' => ['nullable','integer','min:0'],
        ]);

        $data['sort_order'] = $data['sort_order'] ?? 0;

        if ($request->boolean('remove_logo') && $biller->logo && Storage::disk('public')->exists($biller->logo)) {
            Storage::disk('public')->delete($biller->logo);
            $data['logo'] = null;
        }

        if ($request->hasFile('logo_file')) {
            $path = $request->file('logo_file')->store('uploads/billers','public');
            if ($biller->logo && Storage::disk('public')->exists($biller->logo)) {
                Storage::disk('public')->delete($biller->logo);
            }
            $data['logo'] = $path;
        }

        $biller->update($data);
        return redirect(url('/admin/billers'))->with('success', 'Biller updated.');
    }

    public function destroy($id)
    {
        $biller = Biller::findOrFail($id);
        if ($biller->logo && Storage::disk('public')->exists($biller->logo)) {
            Storage::disk('public')->delete($biller->logo);
        }
        $biller->delete();
        return redirect(url('/admin/billers'))->with('success', 'Biller deleted.');
    }

    public function toggle($id)
    {
        $b = Biller::findOrFail($id);
        $b->update(['status' => $b->status === 'active' ? 'inactive' : 'active']);
        return back()->with('success', 'Status toggled.');
    }
}
