<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\BillPayment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class BillPaymentManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-bill-payments');
    }

    /**
     * GET /admin/bill-payments
     * Filters: status=[pending|succeeded|failed], q in account_no/account_name/reference/txn/user/biller
     */
    public function index(Request $request)
    {
        $request->validate([
            'status' => ['nullable','in:pending,succeeded,failed'],
            'q'      => ['nullable','string','max:100'],
        ]);

        $query = BillPayment::with(['user','biller'])->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function($qq) use ($q) {
                $qq->where('account_no','like',"%$q%")
                   ->orWhere('account_name','like',"%$q%")
                   ->orWhere('reference','like',"%$q%")
                   ->orWhere('txn_id','like',"%$q%")
                   ->orWhereHas('user',   fn($uq) => $uq->where('name','like',"%$q%"))
                   ->orWhereHas('biller', fn($bq) => $bq->where('name','like',"%$q%"));
            });
        }

        $payments = $query->paginate(30)->withQueryString();

        return view('admin.bill_payments.index', compact('payments'));
    }

    /**
     * POST /admin/bill-payments/{payment}/succeed
     * Body: txn_id (nullable), reference (nullable)
     * Only allowed from pending.
     */
    public function succeed(BillPayment $payment, Request $request)
    {
        $data = $request->validate([
            'txn_id'    => ['nullable','string','max:100'],
            'reference' => ['nullable','string','max:100'],
        ]);

        DB::transaction(function () use ($payment, $data) {
            $p = BillPayment::whereKey($payment->id)->lockForUpdate()->first();
            if (!$p || $p->status !== 'pending') {
                abort(400, 'Only pending bill payments can be marked succeeded.');
            }

            // (Optional) Wallet debit + logs:
            // $user = $p->user()->lockForUpdate()->first();
            // if (($user->balance ?? 0) < $p->amount) { abort(422,'Insufficient balance'); }
            // $user->balance -= $p->amount; $user->save();
            // Transaction::create([...]);

            $p->update([
                'status'    => 'succeeded',
                'txn_id'    => $data['txn_id'] ?? $p->txn_id,
                'reference' => $data['reference'] ?? $p->reference,
            ]);
        });

        return back()->with('ok', 'Bill payment marked as succeeded.');
    }

    /**
     * POST /admin/bill-payments/{payment}/fail
     * Body: reference (nullable) — failure reason/ref
     * Only allowed from pending.
     */
    public function fail(BillPayment $payment, Request $request)
    {
        $data = $request->validate([
            'reference' => ['nullable','string','max:100'],
        ]);

        DB::transaction(function () use ($payment, $data) {
            $p = BillPayment::whereKey($payment->id)->lockForUpdate()->first();
            if (!$p || $p->status !== 'pending') {
                abort(400, 'Only pending bill payments can be marked failed.');
            }

            $p->update([
                'status'    => 'failed',
                'reference' => $data['reference'] ?? $p->reference,
            ]);
        });

        return back()->with('ok', 'Bill payment marked as failed.');
    }
}
