<?php

namespace App\Http\Controllers\backend;

use App\Http\Controllers\Controller;

use App\Models\Banner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class BannerController extends Controller
{
    public function index(Request $request)
    {
        $q = Banner::query();

        if ($request->filled('active')) {
            $q->where('is_active', (bool)$request->get('active'));
        }

        $banners = $q->orderBy('sort_order')->orderByDesc('id')
                     ->paginate(12)->withQueryString();

        return view('backend.banners.index', compact('banners'));
    }

    public function create()
    {
        return view('backend.banners.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'image_path' => ['nullable','string','max:600'], // URL / text
            'image_file' => ['nullable','image','mimes:jpg,jpeg,png,webp,svg','max:4096'],
            'is_active'  => ['required','in:0,1'],
            'sort_order' => ['nullable','integer','min:0'],
        ]);

        if ($request->hasFile('image_file')) {
            $data['image_path'] = $request->file('image_file')->store('uploads/banners','public');
        }

        $data['sort_order'] = $data['sort_order'] ?? 0;

        Banner::create($data);
        return redirect(url('/admin/banners'))->with('success','Banner created.');
    }

    public function show($id)
    {
        $row = Banner::findOrFail($id);
        return view('backend.banners.show', compact('row'));
    }

    public function edit($id)
    {
        $row = Banner::findOrFail($id);
        return view('backend.banners.edit', compact('row'));
    }

    public function update(Request $request, $id)
    {
        $row = Banner::findOrFail($id);

        $data = $request->validate([
            'image_path' => ['nullable','string','max:600'],
            'image_file' => ['nullable','image','mimes:jpg,jpeg,png,webp,svg','max:4096'],
            'remove_img' => ['nullable','boolean'],
            'is_active'  => ['required','in:0,1'],
            'sort_order' => ['nullable','integer','min:0'],
        ]);

        if ($request->boolean('remove_img') && $row->image_path && Storage::disk('public')->exists($row->image_path)) {
            Storage::disk('public')->delete($row->image_path);
            $data['image_path'] = null;
        }

        if ($request->hasFile('image_file')) {
            $path = $request->file('image_file')->store('uploads/banners','public');
            if ($row->image_path && Storage::disk('public')->exists($row->image_path)) {
                Storage::disk('public')->delete($row->image_path);
            }
            $data['image_path'] = $path;
        }

        $data['sort_order'] = $data['sort_order'] ?? 0;

        $row->update($data);
        return redirect(url('/admin/banners'))->with('success','Banner updated.');
    }

    public function destroy($id)
    {
        $row = Banner::findOrFail($id);
        if ($row->image_path && Storage::disk('public')->exists($row->image_path)) {
            Storage::disk('public')->delete($row->image_path);
        }
        $row->delete();
        return redirect(url('/admin/banners'))->with('success','Banner deleted.');
    }

    public function toggle($id)
    {
        $row = Banner::findOrFail($id);
        $row->update(['is_active' => !$row->is_active]);
        return back()->with('success','Status toggled.');
    }
}
