<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\BankTransfer;
use App\Models\User;
use App\Models\Transaction;   // ✅ Transaction model
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class BankTransferManager extends Controller
{
    public function index(Request $request)
    {
        $request->validate([
            'status' => ['nullable','in:pending,approved,rejected'],
            'q'      => ['nullable','string','max:100'],
        ]);

        $query = BankTransfer::with(['user','bank'])->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function($qq) use ($q) {
                $qq->where('account_name','like',"%$q%")
                   ->orWhere('account_number','like',"%$q%")
                   ->orWhere('branch_name','like',"%$q%")
                   ->orWhereHas('user', fn($uq) => $uq->where('name','like',"%$q%"))
                   ->orWhereHas('bank', fn($bq) => $bq->where('name','like',"%$q%"));
            });
        }

        $transfers = $query->paginate(30)->withQueryString();

        return view('admin.bank_transfers.index', compact('transfers'));
    }

    /**
     * Approve a pending bank transfer:
     * - এখন আর এখানে balance কাটবে না
     * - শুধু status = approved, txn_id set
     * - reward points add হবে
     * - Transaction row তৈরি হবে amount = 0 (audit log)
     */
    public function approve(BankTransfer $transfer, Request $request)
    {
        abort_if($transfer->status !== 'pending', 400, 'Already processed.');

        $data = $request->validate([
            'admin_note' => ['nullable','string','max:2000'],
        ]);

        $awarded = 0;

        try {
            DB::transaction(function () use ($transfer, $data, &$awarded) {

                // Fresh + lock
                $transfer = BankTransfer::whereKey($transfer->id)->lockForUpdate()->firstOrFail();

                if ($transfer->status !== 'pending') {
                    throw new \RuntimeException('ALREADY_PROCESSED');
                }

                /** @var User $user */
                $user = User::whereKey($transfer->user_id)->lockForUpdate()->firstOrFail();

                $amount = (float) $transfer->amount;

                // ❌ আগের কোডে এখানে balance check + debit ছিল
                // ✅ এখন approve সময়ে wallet balance change হবে না

                // Approve + txn_id
                $txnId = 'BNK-'.Str::upper(Str::random(6)).'-'.now()->format('ymdHis');

                $transfer->update([
                    'status'     => 'approved',
                    'admin_note' => $data['admin_note'] ?? $transfer->admin_note,
                    'txn_id'     => $txnId,
                ]);

                // ===== Reward Points (bank_trans_comm %) =====
                $settings = DB::table('reward_settings')
                    ->where('is_active', 1)
                    ->orderByDesc('id')
                    ->first();

                $percent = $settings ? (float) $settings->bank_trans_comm : 0.0;
                $awarded = 0;

                if ($percent > 0 && $amount > 0) {
                    $awarded = (int) floor(($amount * $percent) / 100);

                    if ($awarded > 0) {
                        DB::table('users')
                            ->where('id', $user->id)
                            ->update([
                                'reward_points' => DB::raw('reward_points + '.$awarded),
                            ]);

                        if ($this->tableHasColumn('bank_transfers', 'reward_points_awarded')) {
                            DB::table('bank_transfers')
                                ->where('id', $transfer->id)
                                ->update(['reward_points_awarded' => $awarded]);
                        }
                    }
                }

                // ✅ APPROVE সময়ে Transaction row (amount = 0, শুধু log)
                Transaction::create([
                    'user_id'       => $user->id,
                    'type'          => 'debit',                  // দিক বোঝানোর জন্য (bank এ যাচ্ছে)
                    'amount'        => 0,                        // আসল debit submit সময় হয়েছে
                    'balance_after' => $user->balance,           // এখানে আর change নেই
                    'related_type'  => BankTransfer::class,
                    'related_id'    => $transfer->id,
                    'meta'          => json_encode([
                        'bank_id'        => $transfer->bank_id,
                        'bank_name'      => optional($transfer->bank)->name,
                        'account_name'   => $transfer->account_name,
                        'account_number' => $transfer->account_number,
                        'branch_name'    => $transfer->branch_name,
                        'status'         => 'approved',
                        'txn_id'         => $txnId,
                        'admin_note'     => $transfer->admin_note,
                        'reward_points_awarded' => $awarded,
                        'note'           => 'Bank transfer approved (amount was debited at request time).',
                    ]),
                ]);
            });
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'INSUFFICIENT_BALANCE') {
                // theoretically এখানে আর আসবে না, তবুও handle করে রাখলাম
                return back()->withErrors(['amount' => 'User balance is insufficient to approve this transfer.']);
            }
            if ($e->getMessage() === 'ALREADY_PROCESSED') {
                return back()->with('warning', 'Transfer already processed.');
            }
            throw $e;
        }

        return back()->with('ok', 'Bank transfer approved.');
    }

    /**
     * Reject pending bank transfer
     * - submit এ যে টাকা কাটা হয়েছিল, reject এ সেই টাকা wallet এ ফেরত যাবে
     * - status = rejected
     * - Transaction row create (type=credit, amount = original)
     */
    public function reject(BankTransfer $transfer, Request $request)
    {
        abort_if($transfer->status !== 'pending', 400, 'Already processed.');

        $data = $request->validate([
            'admin_note' => ['required','string','max:2000'],
        ]);

        try {
            DB::transaction(function () use ($transfer, $data) {

                $transfer = BankTransfer::whereKey($transfer->id)->lockForUpdate()->firstOrFail();

                if ($transfer->status !== 'pending') {
                    throw new \RuntimeException('ALREADY_PROCESSED');
                }

                /** @var User $user */
                $user = User::whereKey($transfer->user_id)->lockForUpdate()->firstOrFail();

                $amount = (float) $transfer->amount;

                // ✅ Wallet refund (submit সময় কাটার টাকা ফেরত)
                $newBalance = bcadd($user->balance, $amount, 2);
                $user->update([
                    'balance' => $newBalance,
                ]);

                // Update status + note
                $transfer->update([
                    'status'     => 'rejected',
                    'admin_note' => $data['admin_note'],
                ]);

                // ✅ REJECT সময় Transaction row (credit, refund log)
                Transaction::create([
                    'user_id'       => $user->id,
                    'type'          => 'credit',
                    'amount'        => $amount,
                    'balance_after' => $newBalance,
                    'related_type'  => BankTransfer::class,
                    'related_id'    => $transfer->id,
                    'meta'          => json_encode([
                        'bank_id'        => $transfer->bank_id,
                        'bank_name'      => optional($transfer->bank)->name,
                        'account_name'   => $transfer->account_name,
                        'account_number' => $transfer->account_number,
                        'branch_name'    => $transfer->branch_name,
                        'status'         => 'rejected',
                        'admin_note'     => $data['admin_note'],
                        'note'           => 'Bank transfer request rejected and refunded to wallet.',
                    ]),
                ]);
            });
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'ALREADY_PROCESSED') {
                return back()->with('warning', 'Transfer already processed.');
            }
            throw $e;
        }

        return back()->with('ok', 'Bank transfer rejected.');
    }

    /**
     * Optional helper: check if a table has a column
     */
    protected function tableHasColumn(string $table, string $column): bool
    {
        try {
            /** @var \Illuminate\Database\Schema\Builder $schema */
            $schema = app('db.schema');
            return $schema->hasColumn($table, $column);
        } catch (\Throwable $e) {
            return false;
        }
    }
}
