<?php
// app/Http/Controllers/Backend/AddMoneyCardSubmitManager.php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\AddMoneyCardSubmit;
use App\Models\RewardSetting;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AddMoneyCardSubmitManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-deposits');
    }

    /**
     * GET /admin/card-deposits
     * Filters: status=[pending|approved|rejected], q=search
     */
    public function index(Request $request)
    {
        $request->validate([
            'status' => ['nullable', 'in:pending,approved,rejected'],
            'q'      => ['nullable', 'string', 'max:100'],
        ]);

        $query = AddMoneyCardSubmit::with(['user','card'])->latest();

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function ($qq) use ($q) {
                $qq->where('reference_code', 'like', "%$q%")
                   ->orWhere('holder_name', 'like', "%$q%")
                   ->orWhere('amount', 'like', "%$q%")
                   ->orWhereHas('user', function ($uq) use ($q) {
                       $uq->where('name', 'like', "%$q%");
                   })
                   ->orWhereHas('card', function ($cq) use ($q) {
                       $cq->where('brand', 'like', "%$q%");
                   });
            });
        }

        $submits = $query->paginate(30)->withQueryString();

        return view('admin.add-money-cards.index', compact('submits'));
    }

    /**
     * POST /admin/card-deposits/{submit}/approve
     * Body: note (nullable string)
     */
    public function approve(AddMoneyCardSubmit $submit, Request $request)
    {
        $data = $request->validate([
            'note' => ['nullable', 'string', 'max:500'],
        ]);

        DB::transaction(function () use ($submit, $data) {

            // row lock + fresh read
            $s = AddMoneyCardSubmit::whereKey($submit->id)
                ->lockForUpdate()
                ->with(['user','card'])
                ->first();

            if (! $s || $s->status !== 'pending') {
                abort(400, 'Already processed.');
            }

            $user = $s->user()->lockForUpdate()->first();
            if (! $user) {
                abort(400, 'User missing.');
            }

            // reward config
            $activeSetting = RewardSetting::where('is_active', true)->latest('id')->first();
            $percent       = $activeSetting ? (float) $activeSetting->add_money_comm : 0.0;

            // update status (approved)
            $meta = $s->meta ?? [];
            $meta['admin_note']  = $data['note'] ?? null;
            $meta['approved_at'] = now()->toDateTimeString();

            $s->update([
                'status' => 'approved',
                'meta'   => $meta,
            ]);

            // update wallet balance
            $user->balance = $user->balance + $s->amount;

            // reward points
            $pointsToAdd = (int) floor(($s->amount * $percent) / 100);
            if ($pointsToAdd > 0) {
                $user->reward_points = $user->reward_points + $pointsToAdd;
            }
            $user->save();

            // Transaction log
            Transaction::create([
                'user_id'       => $user->id,
                'type'          => 'credit',
                'amount'        => $s->amount,
                'balance_after' => $user->balance,
                'related_type'  => AddMoneyCardSubmit::class,
                'related_id'    => $s->id,
                'meta'          => [
                    'reference_code' => $s->reference_code,
                    'status'         => $s->status,             // approved
                    'card_brand'     => $s->card?->brand,
                    'trx_type'       => 'Add Money (Card)',
                    'reward_percent' => $percent,
                    'reward_points'  => $pointsToAdd,
                    'note'           => $data['note'] ?? null,
                ],
            ]);
        });

        return back()->with('ok', 'Card add-money request approved & wallet updated.');
    }

    /**
     * POST /admin/card-deposits/{submit}/reject
     * Body: note (required string)
     */
    public function reject(AddMoneyCardSubmit $submit, Request $request)
    {
        $data = $request->validate([
            'note' => ['required', 'string', 'max:500'],
        ]);

        DB::transaction(function () use ($submit, $data) {

            $s = AddMoneyCardSubmit::whereKey($submit->id)->lockForUpdate()->first();

            if (! $s || $s->status !== 'pending') {
                abort(400, 'Already processed.');
            }

            $meta = $s->meta ?? [];
            $meta['admin_note']  = $data['note'];
            $meta['rejected_at'] = now()->toDateTimeString();

            $s->update([
                'status' => 'rejected',
                'meta'   => $meta,
            ]);
        });

        return back()->with('ok', 'Card add-money request rejected.');
    }
}
