<?php

namespace App\Http\Controllers\Backend;

use App\Http\Controllers\Controller;
use App\Models\AddmoneyBank as AddMoneyBank;
use App\Models\AddmoneyBankMethod as AddMoneyBankMethod;
use App\Models\Transaction;
use App\Models\RewardSetting;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AddMoneyBankManager extends Controller
{
    public function __construct()
    {
        // $this->middleware(['auth']);
        // $this->middleware('can:manage-deposits');
    }

    /**
     * GET /admin/bank-deposits
     * Filters: status=[pending|approved|rejected], q=search
     */
    public function index(Request $request)
    {
        $request->validate([
            'status' => ['nullable','in:pending,approved,rejected'],
            'q'      => ['nullable','string','max:100'],
        ]);

        $query = AddMoneyBank::with(['user','method'])->latest(); // method = relation to AddMoneyBankMethod

        if ($request->filled('status')) {
            $query->where('status', $request->string('status')->toString());
        }

        if ($q = $request->string('q')->toString()) {
            $query->where(function($qq) use ($q) {
                $qq->where('reference','like',"%$q%")
                   ->orWhere('account_number','like',"%$q%")
                   ->orWhere('account_holder','like',"%$q%")
                   ->orWhereHas('user', fn($uq) => $uq->where('name','like',"%$q%"));
            });
        }

        $banks = $query->paginate(30)->withQueryString();

        return view('admin.add-money-banks.index', compact('banks'));
    }

    /**
     * POST /admin/bank-deposits/{bank}/approve
     * Body: note (nullable string)
     */
    public function approve(AddMoneyBank $bank, Request $request)
    {
        $data = $request->validate([
            'note' => ['nullable','string','max:500'],
        ]);

        DB::transaction(function () use ($bank, $data) {
            // Lock the row & re-check status
            $b = AddMoneyBank::whereKey($bank->id)->lockForUpdate()->first();

            if (!$b || $b->status !== 'pending') {
                abort(400, 'Already processed.');
            }

            // Lock user
            $user = $b->user()->lockForUpdate()->first();

            // Active reward setting
            $activeSetting = RewardSetting::where('is_active', true)->latest('id')->first();
            $percent = $activeSetting ? (float) $activeSetting->add_money_comm : 0.0;

            // Mark approved early to prevent double approval
            $b->update([
                'status'      => 'approved',
                'approved_at' => now(),
                'admin_note'  => $data['note'] ?? null,
            ]);

            // Wallet credit
            $user->balance = $user->balance + $b->amount;

            // Reward points
            $pointsToAdd = (int) floor(($b->amount * $percent) / 100);
            if ($pointsToAdd > 0) {
                $user->reward_points = $user->reward_points + $pointsToAdd;
            }
            $user->save();

            // Transaction log (polymorphic)
            // আপনি চাইলে associate() ব্যবহার করতে পারেন; নিচে explicit হিসেবে দিলাম।
            Transaction::create([
                'user_id'       => $user->id,
                'type'          => 'credit',
                'amount'        => $b->amount,
                'balance_after' => $user->balance,   // আপনার মাইগ্রেশনের default(0) এর সাথে সঙ্গতিপূর্ণ
                'related_type'  => AddMoneyBank::class,
                'related_id'    => $b->id,
                'meta'          => [
                    'reference'      => $b->reference,
                    'method'         => $b->method?->slug,
                    'reward_percent' => $percent,
                    'reward_points'  => $pointsToAdd,
                    'note'           => $data['note'] ?? null,
                ],
            ]);
        });

        return back()->with('ok', 'Bank deposit approved & reward points updated.');
    }

    /**
     * POST /admin/bank-deposits/{bank}/reject
     * Body: note (required string)
     */
    public function reject(AddMoneyBank $bank, Request $request)
    {
        $data = $request->validate([
            'note' => ['required','string','max:500'],
        ]);

        DB::transaction(function () use ($bank, $data) {
            $b = AddMoneyBank::whereKey($bank->id)->lockForUpdate()->first();

            if (!$b || $b->status !== 'pending') {
                abort(400, 'Already processed.');
            }

            $b->update([
                'status'     => 'rejected',
                'admin_note' => $data['note'],
            ]);
        });

        return back()->with('ok', 'Bank deposit rejected.');
    }
}
