<?php

namespace App\Http\Controllers\User;

use App\Events\MobileTransactionRequested;
use App\Http\Controllers\Controller;
use App\Models\MobileGateway;
use App\Models\MobileTransaction;
use App\Models\Transaction;
use App\Models\BellNotification;
use Illuminate\Broadcasting\BroadcastException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class MobileBankingController extends Controller
{
    public function redirectToDefault()
    {
        return redirect()->route('user.mobile-banking.show', 'bkash');
    }

    public function show(string $code)
    {
        $gateway = MobileGateway::where('code', strtoupper($code))
            ->where('is_active', true)
            ->firstOrFail();

        $gateways = MobileGateway::where('is_active', true)
            ->orderByRaw("FIELD(code, 'BKASH','NAGAD','ROCKET')")
            ->get();

        return view('frontend.mobile_banking.show', compact('gateway','gateways'));
    }

    public function store(Request $request)
    {
        /** @var \App\Models\User $user */
        $user = $request->user();

        $validated = $request->validate([
            'gateway_id'     => ['required','exists:mobile_gateways,id'],
            'account_number' => ['required','string','max:30'],
            'amount'         => ['required','numeric','min:1'],
            'channel'        => ['required','in:agent,personal'],
            'reference'      => ['nullable','string','max:100'],
            'login_pin'      => ['required','digits:4'],
        ], [
            'login_pin.digits' => 'PIN অবশ্যই ৪ ডিজিট হতে হবে।',
        ]);

        $gateway = MobileGateway::findOrFail($validated['gateway_id']);

        // 🔐 PIN verify
        $pin = (string) $request->input('login_pin');

        if (! Hash::check($pin, $user->login_pin)) {
            throw ValidationException::withMessages([
                'login_pin' => 'Incorrect PIN.',
            ]);
        }

        // 🔎 Limit check
        if ($gateway->min_limit > 0 && $validated['amount'] < $gateway->min_limit) {
            return back()
                ->withErrors(['amount' => 'পরিমাণ ন্যূনতম সীমার নিচে।'])
                ->withInput();
        }

        if ($gateway->max_limit > 0 && $validated['amount'] > $gateway->max_limit) {
            return back()
                ->withErrors(['amount' => 'পরিমাণ সর্বোচ্চ সীমার উপরে।'])
                ->withInput();
        }

        // ✅ সবকিছু একসাথে transaction এ করব
        try {
            $mtx = DB::transaction(function () use ($gateway, $validated, $user) {

                /** @var \App\Models\User $lockedUser */
                $lockedUser = \App\Models\User::whereKey($user->id)->lockForUpdate()->firstOrFail();

                $amount = (float) $validated['amount'];

                // 🔎 Wallet balance check (submit সময়েই)
                if (bccomp($lockedUser->balance, $amount, 2) < 0) {
                    throw new \RuntimeException('INSUFFICIENT_BALANCE');
                }

                $commission = $gateway->computeCommission($amount);

                // 💸 Wallet debit (submit সময়)
                $newBalance = bcsub($lockedUser->balance, $amount, 2);
                $lockedUser->update([
                    'balance' => $newBalance,
                ]);

                // 1) mobile transaction (pending)
                $mtx = MobileTransaction::create([
                    'user_id'           => $lockedUser->id,
                    'gateway_id'        => $gateway->id,
                    'account_number'    => $validated['account_number'],
                    'amount'            => $amount,
                    'channel'           => $validated['channel'],
                    'reference'         => $validated['reference'] ?? null,
                    'commission_amount' => $commission,
                    'status'            => 'pending',
                ]);

                // 2) transaction (debit, আসল কাটার লগ)
                Transaction::create([
                    'user_id'       => $lockedUser->id,
                    'type'          => 'debit',
                    'amount'        => $amount,
                    'balance_after' => $newBalance, // ডেবিটের পর balance
                    'related_type'  => MobileTransaction::class,
                    'related_id'    => $mtx->id,
                    'meta'          => json_encode([
                        'gateway_code'   => $gateway->code,
                        'channel'        => $validated['channel'],
                        'account_number' => $validated['account_number'],
                        'reference'      => $validated['reference'] ?? null,
                        'commission'     => (float) $commission,
                        'status'         => 'pending',
                        'note'           => 'Mobile transaction requested and debited from wallet.',
                    ]),
                ]);

                // 3) 🔔 Bell Notification create (mobile_banking)
                $channelLabel = $validated['channel'] === 'agent' ? 'Agent' : 'Personal';
                $refText      = !empty($validated['reference']) 
                    ? " Ref: {$validated['reference']}."
                    : '';

                BellNotification::create([
                    'user_id' => $lockedUser->id,
                    'title'   => 'Mobile Banking Request Submitted',
                    'status'  => 'unread',
                    'type'    => 'mobile_transaction',
                    'amount'  => $amount,
                    'message' => "Your {$channelLabel} {$gateway->code} request of {$amount} TK "
                               . "to account {$validated['account_number']} has been submitted and is pending approval."
                               . " Commission: {$commission} TK."
                               . $refText,
                    'read_at' => null,
                ]);

                return $mtx;
            });
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'INSUFFICIENT_BALANCE') {
                if ($request->wantsJson()) {
                    return response()->json([
                        'ok'      => false,
                        'message' => 'আপনার ওয়ালেট ব্যালেন্স পর্যাপ্ত নয়।',
                    ], 422);
                }

                return back()
                    ->withErrors(['amount' => 'আপনার ওয়ালেট ব্যালেন্স পর্যাপ্ত নয়।'])
                    ->withInput();
            }

            throw $e;
        }

        // যদি উপরের ট্রানজ্যাকশন থেকে JSON রেসপন্স রিটার্ন করে, এখানে এসে $mtx একটা Response হতে পারে,
        // তাই হ্যান্ডেল করে নিচ্ছি
        if ($mtx instanceof \Illuminate\Http\JsonResponse) {
            return $mtx;
        }

        // ✅ DB ঠিকমতো save হওয়ার পর event fire
        try {
            event(new MobileTransactionRequested($mtx));
        } catch (BroadcastException $e) {
            Log::warning('MobileTransactionRequested broadcast failed', [
                'error'              => $e->getMessage(),
                'mobile_transaction' => $mtx->id,
            ]);
        } catch (\Throwable $e) {
            Log::error('MobileTransactionRequested unknown error', [
                'error'              => $e->getMessage(),
                'mobile_transaction' => $mtx->id,
            ]);
        }

        if ($request->wantsJson()) {
            return response()->json([
                'ok'             => true,
                'message'        => 'রিকোয়েস্ট সাবমিট হয়েছে। অ্যাডমিন রিভিউর পর অ্যাপ্রুভ হবে।',
                'transaction_id' => $mtx->id,
                'redirect_url'   => route('user.mobile-banking.success', $mtx),
            ]);
        }

        return redirect()
            ->route('user.mobile-banking.success', $mtx)
            ->with('success', 'রিকোয়েস্ট সাবমিট হয়েছে। অ্যাডমিন রিভিউর পর অ্যাপ্রুভ হবে।');
    }

    public function success(Request $request, MobileTransaction $transaction)
    {
        

        $transaction->load('gateway');

        return view('frontend.mobile_banking.success', [
            'transaction' => $transaction,
        ]);
    }
}
