<?php  

namespace App\Http\Controllers\User;

use App\Events\BankTransferRequested;
use App\Http\Controllers\Controller;
use App\Models\Bank;
use App\Models\BankTransfer;
use App\Models\Transaction;
use App\Models\BellNotification;
use Illuminate\Broadcasting\BroadcastException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Hash;              // ✅ PIN check
use Illuminate\Support\Facades\DB;               // ✅ Wallet debit transaction
use Illuminate\Validation\ValidationException;   // ✅ Wrong PIN error

class BankTransferController extends Controller
{
    public function index(Request $request)
    {
        $q = trim((string) $request->get('q'));

        $banks = Bank::query()
            ->where('is_active', true)
            ->when($q, fn ($qq) => $qq->where('name', 'like', "%$q%"))
            ->orderBy('name')
            ->paginate(36);

        return view('frontend.bank.list', compact('banks', 'q'));
    }

    public function show(Bank $bank)
    {
        abort_if(! $bank->is_active, 404);

        return view('frontend.bank.form', compact('bank'));
    }

    public function store(Request $request, Bank $bank)
    {
        // 1) bank active কিনা
        abort_if(! $bank->is_active, 404);

        // 2) validation + PIN
        $data = $request->validate([
            'account_name'   => ['required', 'string', 'max:120'],
            'account_number' => ['required', 'string', 'max:60'],
            'amount'         => ['required', 'numeric', 'min:1'],
            'branch_name'    => ['nullable', 'string', 'max:120'],
            'login_pin'      => ['required','digits:4'],   // 🔐 4-digit PIN
        ], [
            'login_pin.digits' => 'PIN অবশ্যই ৪ ডিজিট হতে হবে।',
        ]);

        // 3) current user
        $user = $request->user();

        // 4) 🔐 PIN verify
        $pin = (string) $data['login_pin'];

        if (! Hash::check($pin, $user->login_pin)) {
            throw ValidationException::withMessages([
                'login_pin' => 'Incorrect PIN.',
            ]);
        }

        // 5) minimum transaction check
        if ($data['amount'] < (float) $bank->min_transaction) {
            return back()
                ->withErrors([
                    'amount' => 'Minimum Transaction ' . number_format($bank->min_transaction) . ' BDT',
                ])
                ->withInput();
        }

        // 6) সবকিছু একসাথে DB transaction এ
        try {
            /** @var \App\Models\BankTransfer $bt */
            $bt = DB::transaction(function () use ($bank, $data, $user) {

                /** @var \App\Models\User $lockedUser */
                $lockedUser = \App\Models\User::whereKey($user->id)->lockForUpdate()->firstOrFail();

                $amount = (float) $data['amount'];

                // 🔎 Wallet balance check (submit সময়েই)
                if (bccomp($lockedUser->balance, $amount, 2) < 0) {
                    throw new \RuntimeException('INSUFFICIENT_BALANCE');
                }

                // 💸 Wallet debit
                $newBalance = bcsub($lockedUser->balance, $amount, 2);
                $lockedUser->update([
                    'balance' => $newBalance,
                ]);

                // 6.1) BankTransfer row (pending)
                $bt = BankTransfer::create([
                    'user_id'        => $lockedUser->id,
                    'bank_id'        => $bank->id,
                    'account_name'   => $data['account_name'],
                    'account_number' => $data['account_number'],
                    'amount'         => $amount,
                    'branch_name'    => $data['branch_name'] ?? null,
                    'status'         => 'pending',
                ]);

                // 6.2) Transaction (আসল debit log)
                Transaction::create([
                    'user_id'       => $lockedUser->id,
                    'type'          => 'debit',
                    'amount'        => $amount,
                    'balance_after' => $newBalance,
                    'related_type'  => BankTransfer::class,
                    'related_id'    => $bt->id,
                    'meta'          => json_encode([
                        'bank_id'        => $bank->id,
                        'bank_name'      => $bank->name,
                        'account_name'   => $data['account_name'],
                        'account_number' => $data['account_number'],
                        'branch_name'    => $data['branch_name'] ?? null,
                        'status'         => 'pending',
                        'note'           => 'Bank transfer requested and debited from wallet.',
                    ]),
                ]);

                // 6.3) 🔔 Bell Notification
                $branchText = !empty($data['branch_name'])
                    ? " (Branch: {$data['branch_name']})"
                    : '';

                BellNotification::create([
                    'user_id' => $lockedUser->id,
                    'title'   => 'Bank Transfer Request Submitted',
                    'status'  => 'unread',
                    'type'    => 'bank_transfer',
                    'amount'  => $amount,
                    'message' => "Your bank transfer request of {$amount} BDT to {$bank->name} "
                               . "account {$data['account_number']}{$branchText} has been submitted and is waiting for approval.",
                    'read_at' => null,
                ]);

                return $bt;
            });
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'INSUFFICIENT_BALANCE') {
                return back()
                    ->withErrors(['amount' => 'আপনার ওয়ালেট ব্যালেন্স পর্যাপ্ত নয়।'])
                    ->withInput();
            }

            throw $e;
        }

        // 7) সব save হওয়ার পর event fire
        try {
            event(new BankTransferRequested($bt));
        } catch (BroadcastException $e) {
            Log::warning('BankTransferRequested broadcast failed', [
                'error'            => $e->getMessage(),
                'bank_transfer_id' => $bt->id,
            ]);
        } catch (\Throwable $e) {
            Log::error('BankTransferRequested unknown error', [
                'error'            => $e->getMessage(),
                'bank_transfer_id' => $bt->id,
            ]);
        }

        // 8) normal form submit → আলাদা success page
        return redirect()
            ->route('user.bank.transfer.success', $bt)
            ->with('success', 'Bank transfer request submitted. Waiting for approval.');
    }

    /**
     * Bank Transfer success page
     */
    public function success(Request $request, BankTransfer $transfer)
    {
        

        $transfer->load('bank');

        return view('frontend.bank.success', [
            'transfer' => $transfer,
        ]);
    }
}
