<?php

// app/Http/Controllers/UploadAppController.php

namespace App\Http\Controllers;

use App\Models\UploadApp;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class UploadAppController extends Controller
{
    public function index()
    {
        $apps = UploadApp::latest()->paginate(15);

        return view('admin.upload_apps.index', compact('apps'));
    }

    public function create()
    {
        return view('admin.upload_apps.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            // সব extension allow করতে চাইলে শুধু file + max
            'upload_apk' => 'required|file|max:51200', // ~50MB
            // যদি শুধু apk/zip রাখতে চাও:
            // 'upload_apk' => 'required|file|mimes:apk,zip|max:51200',
        ]);

        if ($request->hasFile('upload_apk')) {
            $file = $request->file('upload_apk');

            // আসল নাম + extension
            $originalName = $file->getClientOriginalName();           // eg: nhbPbzjyMu75KHUQjgmgnDotmAr7kCQmzte99dXM.zip
            $extension    = $file->getClientOriginalExtension();      // eg: zip
            $baseName     = pathinfo($originalName, PATHINFO_FILENAME); // name without extension

            // নাম clean + টাইমস্ট্যাম্প + আসল extension
            $safeBaseName = Str::slug($baseName) ?: 'app-file';
            $finalName    = $safeBaseName . '-' . time() . '.' . $extension;
            // এখানে extension একদমই change হচ্ছে না 🔥

            // storage/app/public/upload_apps এ সেভ হবে
            $path = $file->storeAs('upload_apps', $finalName, 'public');

            UploadApp::create([
                'upload_apk' => $path,
            ]);

            return redirect()
                ->route('upload-apps.index')
                ->with('success', 'App uploaded successfully.');
        }

        return back()->withErrors(['upload_apk' => 'File upload failed.']);
    }

    public function edit(UploadApp $app)
    {
        $uploadApp = $app;
        return view('admin.upload_apps.edit', compact('uploadApp'));
    }

    public function update(Request $request, UploadApp $app)
    {
        $request->validate([
            'upload_apk' => 'nullable|file|max:51200',
            // বা শুধু apk/zip চাইলে:
            // 'upload_apk' => 'nullable|file|mimes:apk,zip|max:51200',
        ]);

        if ($request->hasFile('upload_apk')) {
            $file = $request->file('upload_apk');

            // পুরনো ফাইল থাকলে delete
            if ($app->upload_apk && Storage::disk('public')->exists($app->upload_apk)) {
                Storage::disk('public')->delete($app->upload_apk);
            }

            $originalName = $file->getClientOriginalName();
            $extension    = $file->getClientOriginalExtension();
            $baseName     = pathinfo($originalName, PATHINFO_FILENAME);
            $safeBaseName = Str::slug($baseName) ?: 'app-file';
            $finalName    = $safeBaseName . '-' . time() . '.' . $extension;

            $path = $file->storeAs('upload_apps', $finalName, 'public');

            $app->upload_apk = $path;
            $app->save();
        }

        return redirect()
            ->route('upload-apps.index')
            ->with('success', 'App updated successfully.');
    }

    public function destroy(UploadApp $app)
    {
        if ($app->upload_apk && Storage::disk('public')->exists($app->upload_apk)) {
            Storage::disk('public')->delete($app->upload_apk);
        }

        $app->delete();

        return redirect()
            ->route('upload-apps.index')
            ->with('success', 'App deleted successfully.');
    }
}
