<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class StatementController extends Controller
{
    /**
     * স্টেটমেন্ট ফর্ম দেখানো
     */
    public function showForm()
    {
        $user = Auth::user();

        return view('frontend.statement.form', compact('user'));
    }

    /**
     * ফর্ম সাবমিট → পাসওয়ার্ড চেক + ট্রানজেকশন লোড
     */
    public function generate(Request $request)
    {
        $user = Auth::user();

        $data = $request->validate([
            'from_date' => ['required', 'date'],
            'to_date'   => ['required', 'date', 'after_or_equal:from_date'],
            'password'  => ['required', 'string'],
        ]);

        // পাসওয়ার্ড যাচাই (login password)
        if (! Hash::check($data['password'], $user->password)) {
            return back()
                ->withErrors(['password' => 'পাসওয়ার্ড সঠিক নয়।'])
                ->withInput($request->except('password'));
        }

        // ✅ তারিখগুলো Asia/Dhaka ধরে নিচ্ছি
        $from = Carbon::parse($data['from_date'], 'Asia/Dhaka')->startOfDay();
        $to   = Carbon::parse($data['to_date'], 'Asia/Dhaka')->endOfDay();

        // ✅ নির্বাচিত রেঞ্জের ট্রানজেকশন (latest first)
        $transactions = Transaction::where('user_id', $user->id)
            ->whereBetween('created_at', [$from, $to])
            ->orderBy('created_at', 'desc')   // ⬅️ latest উপরে
            ->get();

        // ওপেনিং ব্যালেন্স (from_date-এর আগে সর্বশেষ balance_after)
        $openingBalance = Transaction::where('user_id', $user->id)
            ->where('created_at', '<', $from)
            ->orderBy('created_at', 'desc')
            ->value('balance_after') ?? 0;

        return view('frontend.statement.view', [
            'user'            => $user,
            'transactions'    => $transactions,
            'from'            => $from,
            'to'              => $to,
            'openingBalance'  => $openingBalance,
        ]);
    }

    public function downloadPdf(Transaction $transaction)
    {
        $user = auth()->user();

        $meta = is_array($transaction->meta)
            ? $transaction->meta
            : (json_decode($transaction->meta ?? '[]', true) ?? []);

        $pdf = \PDF::loadView('frontend.statement.pdf', [
            'transaction' => $transaction,
            'meta'        => $meta,
            'user'        => $user,
        ]);

        return $pdf->download('statement-'.$transaction->id.'.pdf');
    }
}
