<?php

namespace App\Http\Controllers;

use App\Events\SendMoneyRequested;
use App\Http\Requests\SendMoneyRequest;
use App\Models\User;
use App\Models\Transfer;
use App\Models\BellNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\ValidationException;

class SendMoneyController extends Controller
{
    public function searchUser(Request $request)
    {
        $request->validate([
            'phone' => ['required','regex:/^\+?\d{10,15}$/'],
        ]);

        $user = User::byPhone($request->phone)->first();

        if (! $user) {
            return response()->json(['found' => false], 404);
        }

        return response()->json([
            'found' => true,
            'user'  => [
                'id'    => $user->id,
                'name'  => $user->name,
                'phone' => $user->phone,
            ],
        ]);
    }

    public function create()
    {
        return view('frontend.wallet.send');
    }

    /**
     * Send money request → only creates a pending Transfer.
     * No balance change / transactions here.
     */
    public function store(SendMoneyRequest $request)
    {
        $from    = $request->user();
        $toPhone = (string) $request->input('to_phone');
        $amount  = (float) $request->input('amount');
        $note    = $request->input('note');
        $key     = $request->input('idempotency_key');

        // 0️⃣ PIN validate (SERVER-SIDE)
        $pin = (string) $request->input('login_pin');

        if (! Hash::check($pin, $from->login_pin)) { // column নাম মিলিয়ে নিও
            throw ValidationException::withMessages([
                'login_pin' => 'Incorrect PIN.',
            ]);
        }

        // ---- Basic validations ----
        if ($amount <= 0) {
            throw ValidationException::withMessages([
                'amount' => 'Amount must be greater than 0.',
            ]);
        }

        $to = User::byPhone($toPhone)->first();
        if (! $to) {
            throw ValidationException::withMessages([
                'to_phone' => 'Recipient not found.',
            ]);
        }

        if ($from->id === $to->id) {
            throw ValidationException::withMessages([
                'to_phone' => 'You cannot send money to yourself.',
            ]);
        }

        // Optional: ensure user currently has enough balance
        if ($from->balance < $amount) {
            throw ValidationException::withMessages([
                'amount' => 'Insufficient balance.',
            ]);
        }

        // Idempotency key check
        if ($key) {
            $existing = Transfer::where('idempotency_key', $key)->first();

            if ($existing) {
                $msg = $existing->status === 'pending'
                    ? 'Your previous send money request is already pending for approval.'
                    : 'This send money request was already processed.';

                $successUrl = route('wallet.send.success', $existing);

                // যদি কখনও API থেকে use করো, তখন JSON পাবে
                if ($request->wantsJson()) {
                    return response()->json([
                        'ok'           => true,
                        'message'      => $msg,
                        'transfer_id'  => $existing->id,
                        'status'       => $existing->status,
                        'redirect_url' => $successUrl,
                    ]);
                }

                // 🟢 এখন স্বাভাবিক redirect
                return redirect()
                    ->to($successUrl)
                    ->with('success', $msg);
            }
        }

        // ---- Main: only create Transfer row, status = pending ----
        $transfer = DB::transaction(function () use ($from, $to, $amount, $note, $key) {

            // 1️⃣ Transfer create
            $transfer = Transfer::create([
                'from_user_id'    => $from->id,
                'to_user_id'      => $to->id,
                'amount'          => $amount,
                'note'            => $note,
                'idempotency_key' => $key,
                'status'          => 'pending',
            ])->fresh(['fromUser','toUser']);

            // 2️⃣ Bell notification (for sender)
            BellNotification::create([
                'user_id' => $from->id,
                'title'   => 'Send Money Request Submitted',
                'status'  => 'unread',
                'type'    => 'send_money',
                'amount'  => $amount,
                'message' => "Your send money request of {$amount} TK to {$to->name} ({$to->phone}) has been submitted and is waiting for admin approval.",
                'read_at' => null,
            ]);

            return $transfer;
        });

        // Event fire
        event(new SendMoneyRequested($transfer));

        // ✅ success page URL
        $successUrl = route('wallet.send.success', $transfer);

        // আবারও, যদি API client কখনও লাগে:
        if ($request->wantsJson()) {
            return response()->json([
                'ok'           => true,
                'message'      => 'Send money request submitted. Waiting for admin approval.',
                'transfer_id'  => $transfer->id,
                'status'       => $transfer->status,
                'to'           => $transfer->toUser->only('id','name','phone'),
                'amount'       => (float) $transfer->amount,
                'redirect_url' => $successUrl,
            ]);
        }

        // 🟢 Normal form submit হলে এখানে সরাসরি success page এ যাবে
        return redirect()
            ->to($successUrl)
            ->with('success', 'Send money request submitted. Waiting for admin approval.');
    }

    /**
     * Success page (after creating send money request)
     */
    public function success(Request $request, Transfer $transfer)
    {
        if ($transfer->from_user_id !== $request->user()->id) {
            abort(403);
        }

        return view('frontend.wallet.send_success', [
            'transfer' => $transfer,
        ]);
    }
}
