<?php  

namespace App\Http\Controllers;

use App\Events\RewardConverted;            // Event
use App\Models\RewardConversion;
use App\Models\User;
use App\Models\Transaction;
use App\Models\BellNotification;          // Notification model
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;       // 🔐 PIN check
use Illuminate\Validation\ValidationException; // 🔐 Wrong PIN

class RewardConversionController extends Controller
{
    private int $RATE_POINTS = 100;   // 100 points =
    private float $RATE_AMOUNT = 1.0; // ৳1.00

    public function index(Request $request)
    {
        $user       = $request->user();
        $ratePoints = $this->RATE_POINTS;
        $rateAmount = $this->RATE_AMOUNT;

        return view('frontend.rewards.convert', compact('user','ratePoints','rateAmount'));
    }

    public function store(Request $request)
    {
        $ratePoints = $this->RATE_POINTS;
        $rateAmount = $this->RATE_AMOUNT;

        $data = $request->validate([
            'points'    => ['required','integer','min:'.$ratePoints],
            'ref'       => ['nullable','string','max:80'],
            'login_pin' => ['required','digits:4'], // 🔐 ৪ ডিজিট PIN
        ], [
            'login_pin.digits' => 'PIN অবশ্যই ৪ ডিজিট হতে হবে।',
        ]);

        /** @var \App\Models\User $user */
        $user   = $request->user();
        $userId = $user->id;

        // 🔐 PIN verify
        $pin = (string) $data['login_pin'];

        // users টেবিলে ধরে নিচ্ছি bcrypt করা কলাম হলো login_pin
        if (! Hash::check($pin, $user->login_pin)) {
            throw ValidationException::withMessages([
                'login_pin' => 'Incorrect PIN.',
            ]);
        }

        // পয়েন্ট অবশ্যই ratePoints এর গুণিতক হতে হবে
        if ($data['points'] % $ratePoints !== 0) {
            return back()
                ->withErrors(['points' => 'পয়েন্ট অবশ্যই '.$ratePoints.'-এর গুণিতক হতে হবে.'])
                ->withInput();
        }

        try {
            /** @var \App\Models\RewardConversion $conversion */
            $conversion = DB::transaction(function () use ($userId, $data, $ratePoints, $rateAmount) {
                /** @var User $user */
                $user = User::whereKey($userId)->lockForUpdate()->first();

                // 🔒 পর্যাপ্ত পয়েন্ট না থাকলে error
                if ($user->reward_points < $data['points']) {
                    throw new \RuntimeException('INSUFFICIENT_POINTS');
                }

                // 🎯 টাকার পরিমাণ হিসাব
                $steps  = intdiv($data['points'], $ratePoints);
                $amount = round($steps * $rateAmount, 2); // e.g. 100 → 1.00

                // ✅ Reward points কমাও, ব্যালেন্স বাড়াও
                $user->decrement('reward_points', (int) $data['points']);
                $user->increment('balance', $amount);

                // ✅ RewardConversion রেকর্ড তৈরি করো
                $conversion = RewardConversion::create([
                    'user_id'     => $user->id,
                    'points'      => (int) $data['points'],
                    'amount'      => $amount,
                    'rate_points' => $ratePoints,
                    'rate_amount' => $rateAmount,
                    'ref'         => $data['ref'] ?? null,
                ]);

                // ✅ Transaction রেকর্ড তৈরি করো (credit)
                Transaction::create([
                    'user_id'       => $user->id,
                    'type'          => 'credit', // ব্যালেন্স বাড়ছে
                    'amount'        => $amount,
                    'balance_after' => $user->fresh()->balance,
                    'related_type'  => RewardConversion::class,
                    'related_id'    => $conversion->id,
                    'meta'          => json_encode([
                        'points_converted' => (int) $data['points'],
                        'rate'             => "{$ratePoints} pts = ৳{$rateAmount}",
                        'ref'              => $data['ref'] ?? null,
                    ]),
                ]);

                // 🔔 ✅ Bell Notification create (safe ref handling)
                $refText = (isset($data['ref']) && $data['ref'] !== null && $data['ref'] !== '')
                    ? " Ref: {$data['ref']}."
                    : '';

                BellNotification::create([
                    'user_id' => $user->id,
                    'title'   => 'Reward Points Converted',
                    'status'  => 'unread',
                    'type'    => 'reward_conversion',
                    'amount'  => $amount,
                    'message' => "You converted {$data['points']} reward points into {$amount} TK." . $refText,
                    'read_at' => null,
                ]);

                // ⬇️ transaction শেষে বাইরে ব্যবহার করব
                return $conversion;
            });

        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'INSUFFICIENT_POINTS') {
                return back()
                    ->withErrors(['points' => 'আপনার কাছে পর্যাপ্ত পয়েন্ট নেই.'])
                    ->withInput();
            }
            throw $e;
        }

        // 🔔 Transaction success → এখন event fire করো
        event(new RewardConverted($conversion));

        // JSON client হলে success + redirect URL
        if ($request->wantsJson()) {
            return response()->json([
                'ok'           => true,
                'message'      => 'পয়েন্ট সফলভাবে কনভার্ট হয়েছে এবং ট্রানজ্যাকশন যুক্ত হয়েছে।',
                'conversion_id'=> $conversion->id,
                'redirect_url' => route('rewards.convert.success', $conversion),
            ]);
        }

        // 👉 এখন আলাদা success পেজে redirect
        return redirect()
            ->route('rewards.convert.success', $conversion)
            ->with('success','পয়েন্ট সফলভাবে কনভার্ট হয়েছে এবং ট্রানজ্যাকশন যুক্ত হয়েছে।');
    }

    /**
     * Success page (Reward conversion summary)
     */
    public function success(Request $request, RewardConversion $conversion)
    {
        // নিজের conversion না হলে দেখাবো না
        // if ($conversion->user_id !== $request->user()->id) {
        //     abort(403);
        // }

        return view('frontend.rewards.convert_success', [
            'conversion' => $conversion,
        ]);
    }
}
