<?php

namespace App\Http\Controllers;

use App\Models\Recharge;
use App\Models\User;
use App\Models\Transaction;
use App\Models\BellNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Events\RechargeRequested;
use Illuminate\Support\Facades\Hash;              // ✅ PIN check
use Illuminate\Validation\ValidationException;   // ✅ PIN ভুল হলে error

class RechargeController extends Controller
{
    public function index(Request $request)
    {
        return view('frontend.recharge.index');
    }

    public function store(Request $request)
    {
        /** @var User $user */
        $user = $request->user();

        // ✅ validation + PIN field
        $data = $request->validate([
            'service_type' => ['required','in:prepaid,postpaid'],
            'msisdn'       => ['required','regex:/^\+?\d{10,15}$/'],
            'amount'       => ['required','numeric','min:10','max:5000'],
            'reference'    => ['nullable','string','max:80'],
            'login_pin'    => ['required','digits:4'], // 🔐 4 digit PIN
        ], [
            'msisdn.regex'    => 'সঠিক মোবাইল নম্বর দিন (10–15 ডিজিট, + ঐচ্ছিক)।',
            'amount.min'      => 'ন্যূনতম রিচার্জ 10 টাকা।',
            'login_pin.digits'=> 'PIN অবশ্যই ৪ ডিজিট হতে হবে।',
        ]);

        // 🔐 PIN verify (send money / bill payment এর মতই)
        $pin = (string) $request->input('login_pin');

        if (! Hash::check($pin, $user->login_pin)) {    // column নাম পরিবর্তন করলে এখানে ঠিক করে নিও
            throw ValidationException::withMessages([
                'login_pin' => 'Incorrect PIN.',
            ]);
        }

        $userId = $user->id;

        try {
            $result = DB::transaction(function () use ($userId, $data) {
                /** @var User $user */
                $user = User::whereKey($userId)->lockForUpdate()->first();

                // 1️⃣ টাকা যথেষ্ট নয় → failed রেকর্ড তৈরি
                if (bccomp($user->balance, $data['amount'], 2) < 0) {
                    Recharge::create([
                        'user_id'      => $user->id,
                        'service_type' => $data['service_type'],
                        'msisdn'       => preg_replace('/\D/','', $data['msisdn']),
                        'amount'       => $data['amount'],
                        'reference'    => $data['reference'] ?? null,
                        'status'       => 'failed',
                        'txn_id'       => null,
                    ]);

                    throw new \RuntimeException('INSUFFICIENT_BALANCE');
                }

                // 2️⃣ ব্যালেন্স ডেবিট
                $user->update([
                    'balance' => bcsub($user->balance, $data['amount'], 2),
                ]);

                // 3️⃣ Recharge রেকর্ড তৈরি (pending)
                $recharge = Recharge::create([
                    'user_id'      => $user->id,
                    'service_type' => $data['service_type'],
                    'msisdn'       => preg_replace('/\D/','', $data['msisdn']),
                    'amount'       => $data['amount'],
                    'reference'    => $data['reference'] ?? null,
                    'status'       => 'pending',
                    'txn_id'       => 'RCG-' . Str::upper(Str::random(6)) . '-' . now()->format('ymdHis'),
                ]);

                // 4️⃣ Reward Points হিসাব
                $settings = DB::table('reward_settings')
                    ->where('is_active', 1)
                    ->orderByDesc('id')
                    ->first();

                $percent = $settings ? (float) $settings->mobile_recharge_comm : 0.0;
                $awarded = 0;

                if ($percent > 0) {
                    $awarded = (int) floor(($data['amount'] * $percent) / 100);
                    if ($awarded > 0) {
                        $user->increment('reward_points', $awarded);
                    }
                }

                // 5️⃣ reward_points_awarded কলাম থাকলে আপডেট করো
                if ($this->tableHasColumn('recharges', 'reward_points_awarded')) {
                    DB::table('recharges')->where('id', $recharge->id)->update([
                        'reward_points_awarded' => $awarded,
                    ]);
                }

                // 6️⃣ Transaction রেকর্ড তৈরি
                Transaction::create([
                    'user_id'       => $user->id,
                    'type'          => 'debit',
                    'amount'        => $data['amount'],
                    'balance_after' => $user->fresh()->balance,
                    'related_type'  => Recharge::class,
                    'related_id'    => $recharge->id,
                    'meta'          => json_encode([
                        'service_type' => $data['service_type'],
                        'msisdn'       => $data['msisdn'],
                        'txn_id'       => $recharge->txn_id,
                        'reward_points_awarded' => $awarded,
                    ]),
                ]);

                // 7️⃣ 🔔 Bell Notification
                $cleanMsisdn   = preg_replace('/\D/', '', $data['msisdn']);
                $serviceLabel  = $data['service_type'] === 'prepaid' ? 'Prepaid' : 'Postpaid';

                BellNotification::create([
                    'user_id' => $user->id,
                    'title'   => 'Recharge Request Submitted',
                    'status'  => 'unread',
                    'type'    => 'recharge',
                    'amount'  => $data['amount'],
                    'message' => "Your {$serviceLabel} recharge request of {$data['amount']} TK "
                               . "for {$cleanMsisdn} has been submitted and is waiting for approval. "
                               . "TXN: {$recharge->txn_id}"
                               . ($awarded > 0 ? " | Reward: +{$awarded} pts" : ''),
                    'read_at' => null,
                ]);

                return [
                    'recharge'     => $recharge,
                    'txn_id'       => $recharge->txn_id,
                    'awarded'      => $awarded,
                    'new_points'   => (int) $user->reward_points,
                    'new_balance'  => (float) $user->fresh()->balance,
                ];
            });
        } catch (\RuntimeException $e) {
            if ($e->getMessage() === 'INSUFFICIENT_BALANCE') {
                if ($request->wantsJson()) {
                    return response()->json([
                        'ok'      => false,
                        'message' => 'আপনার ব্যালেন্স যথেষ্ট নয়।',
                    ], 422);
                }

                return back()
                    ->withErrors(['amount' => 'আপনার ব্যালেন্স যথেষ্ট নয়।'])
                    ->withInput();
            }

            throw $e;
        }

        /** @var Recharge $recharge */
        $recharge = $result['recharge'];
        event(new RechargeRequested($recharge));

        if ($request->wantsJson()) {
            return response()->json([
                'ok'             => true,
                'message'        => 'রিচার্জ সফল হয়েছে।',
                'txn_id'         => $result['txn_id'],
                'reward_awarded' => $result['awarded'],
                'reward_points'  => $result['new_points'],
                'balance'        => $result['new_balance'],
            ]);
        }

        // normal form submit → success page
        return redirect()
            ->route('recharge.success', $recharge)
            ->with('success', 'রিচার্জ সফল হয়েছে।');
    }

    public function success(Request $request, Recharge $recharge)
    {
        // if ($recharge->user_id !== $request->user()->id) {
        //     abort(403);
        // }

        return view('frontend.recharge.success', compact('recharge'));
    }

    protected function tableHasColumn(string $table, string $column): bool
    {
        try {
            $schema = app('db.schema');
            return $schema->hasColumn($table, $column);
        } catch (\Throwable $e) {
            return false;
        }
    }
}
