<?php 

namespace App\Http\Controllers\Loan;

use App\Events\LoanApplicationSubmitted;
use App\Http\Controllers\Controller;
use App\Models\LoanApplication;
use App\Models\LoanSetting;
use App\Models\Transaction;
use App\Models\User;
use App\Models\BellNotification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Hash;              // ✅ PIN check
use Illuminate\Validation\ValidationException;   // ✅ nice error

class ApplyController extends Controller
{
    /**
     * Loan apply form
     */
    public function create(Request $request)
    {
        $settings = LoanSetting::current();
        $user     = $request->user();
        $turnover = $this->userTurnover($user->id);
        $required = (float) $settings->eligibility_min_turnover;
        $eligible = $turnover >= $required;

        return view('frontend.loans.apply', compact(
            'settings', 'turnover', 'required', 'eligible', 'user'
        ));
    }

    /**
     * Store loan application + create a Transaction (credit, pending reference)
     */
    public function store(Request $request)
    {
        $settings = LoanSetting::current();
        $user     = $request->user();
        $turnover = $this->userTurnover($user->id);
        $required = (float) $settings->eligibility_min_turnover;

        // ✅ eligibility check
        if ($turnover < $required) {
            return back()->withErrors([
                'loan_amount' => 'আপনি এখনও লোনের জন্য যোগ্য নন। ন্যূনতম ৳' . number_format($required, 2) . ' লেনদেন প্রয়োজন।'
            ])->withInput();
        }

        // ✅ validation (PIN সহ)
        $data = $request->validate([
            'loan_amount' => ['required', 'integer', 'min:1000'],
            'loan_period' => ['required', 'integer', 'min:1'],
            'document'    => ['nullable', 'image', 'max:2048'],
            'login_pin'   => ['required', 'digits:4'],  // 🔐 PIN (4 digits)
        ], [
            'login_pin.digits' => 'PIN অবশ্যই ৪ ডিজিট হতে হবে।',
        ]);

        // 🔐 PIN verify (same pattern as অন্য পেজগুলো)
        $pin = (string) $request->input('login_pin');

        // users টেবিলে যদি bcrypt করা login_pin থাকে:
        if (! Hash::check($pin, $user->login_pin)) {   // ← column name লাগলে বদলাও
            throw ValidationException::withMessages([
                'login_pin' => 'Incorrect PIN.',
            ]);
        }

        $rate   = (float) $settings->interest_rate; // %
        $amount = (int) $data['loan_amount'];
        $period = (int) $data['loan_period'];       // months
        $total  = (int) round($amount + ($amount * ($rate / 100) * ($period / 12)));

        $path = null;
        if ($request->hasFile('document')) {
            $path = $request->file('document')->store('loans', 'public');
        }

        // ✅ Everything atomic
        $loan = DB::transaction(function () use ($user, $amount, $period, $rate, $total, $path) {

            // 1) LoanApplication (pending)
            $loan = LoanApplication::create([
                'user_id'         => $user->id,
                'loan_amount'     => $amount,
                'loan_period'     => $period,
                'interest_rate'   => $rate,
                'estimated_total' => $total,
                'image_path'      => $path ? '/storage/' . $path : null,
                'status'          => 'pending',
            ]);

            // 2) Transaction reference (balance change এখনও না)
            Transaction::create([
                'user_id'       => $user->id,
                'type'          => 'credit',                   // loan inflow (future)
                'amount'        => $amount,
                'balance_after' => $user->balance,             // not posted yet
                'related_type'  => LoanApplication::class,
                'related_id'    => $loan->id,
                'meta'          => json_encode([
                    'status'          => 'pending',
                    'loan_period'     => $period,
                    'interest_rate'   => $rate,
                    'estimated_total' => $total,
                    'note'            => 'Loan application submitted; disbursement pending.',
                ]),
            ]);

            // 3) 🔔 Bell Notification
            BellNotification::create([
                'user_id' => $user->id,
                'title'   => 'Loan Application Submitted',
                'status'  => 'unread',
                'type'    => 'loan_application',
                'amount'  => $amount,
                'message' => "Your loan application of {$amount} TK for {$period} month(s) at {$rate}% interest "
                           . "(estimated total {$total} TK) has been submitted and is waiting for review.",
                'read_at' => null,
            ]);

            return $loan;
        });

        // ✅ event
        event(new LoanApplicationSubmitted($loan));

        // JSON ক্লায়েন্ট হলে
        if ($request->wantsJson()) {
            return response()->json([
                'ok'        => true,
                'message'   => 'Loan application submitted.',
                'loan_id'   => $loan->id,
                'status'    => $loan->status,
                'redirect'  => route('loans.success', $loan),
            ]);
        }

        // 🟢 normal form submit → success page
        return redirect()
            ->route('loans.success', $loan)
            ->with('success', 'Loan application submitted and transaction logged.');
    }

    /**
     * Simple loan details (পুরনো ডিটেইল পেজ রাখতে চাইলে)
     */
    public function show(LoanApplication $loan)
    {
        $loan->load('repayments');
        return view('frontend.loans.show', compact('loan'));
    }

    /**
     * ✅ Success page (same style কার্ড)
     */
    public function success(Request $request, LoanApplication $loan)
    {
        // নিজের লোন না হলে দেখাতে দিবে না
        // if ($loan->user_id !== $request->user()->id) {
        //     abort(403);
        // }

        return view('frontend.loans.success', compact('loan'));
    }

    /**
     * OPTIONAL: Approve/disburse method (unchanged)
     */
    public function approve(Request $request, LoanApplication $loan)
    {
        if ($loan->status !== 'pending') {
            return back()->withErrors(['loan' => 'This loan is not pending.']);
        }

        DB::transaction(function () use ($loan) {
            /** @var User $user */
            $user = User::lockForUpdate()->findOrFail($loan->user_id);

            // 1) Mark approved
            $loan->update(['status' => 'approved']);
            $loan->refresh();

            // 2) Disburse: increase balance
            $user->increment('balance', (float) $loan->loan_amount);

            // 3) Final transaction with posted balance
            Transaction::create([
                'user_id'       => $user->id,
                'type'          => 'credit',
                'amount'        => (float) $loan->loan_amount,
                'balance_after' => $user->fresh()->balance,
                'related_type'  => LoanApplication::class,
                'related_id'    => $loan->id,
                'meta'          => json_encode([
                    'status'          => 'approved_disbursed',
                    'loan_period'     => (int) $loan->loan_period,
                    'interest_rate'   => (float) $loan->interest_rate,
                    'estimated_total' => (float) $loan->estimated_total,
                    'note'            => 'Loan disbursed to wallet.',
                ]),
            ]);

            // 4) 🔔 Bell Notification
            BellNotification::create([
                'user_id' => $user->id,
                'title'   => 'Loan Approved & Disbursed',
                'status'  => 'unread',
                'type'    => 'loan_approved',
                'amount'  => (float) $loan->loan_amount,
                'message' => "Your loan of {$loan->loan_amount} TK has been approved and added to your wallet balance. "
                           . "Total payable (with interest): {$loan->estimated_total} TK.",
                'read_at' => null,
            ]);
        });

        return back()->with('success', 'Loan approved and disbursed.');
    }

    /**
     * ইউজারের টার্নওভার ক্যালকুলেশন—আগের মতোই
     */
    private function userTurnover(int $userId): float
    {
        $sum = 0.0;

        // Purchases (offers) — succeeded
        if (Schema::hasTable('purchases')) {
            $sum += (float) DB::table('purchases')
                ->where('user_id', $userId)
                ->where('status', 'succeeded')
                ->sum('price');
        }

        // Recharges — succeeded
        if (Schema::hasTable('recharges')) {
            $sum += (float) DB::table('recharges')
                ->where('user_id', $userId)
                ->where('status', 'succeeded')
                ->sum('amount');
        }

        // Bill payments — succeeded
        if (Schema::hasTable('bill_payments')) {
            $sum += (float) DB::table('bill_payments')
                ->where('user_id', $userId)
                ->where('status', 'succeeded')
                ->sum('amount');
        }

        // Deposits — approved
        if (Schema::hasTable('deposits')) {
            $sum += (float) DB::table('deposits')
                ->where('user_id', $userId)
                ->where('status', 'approved')
                ->sum('amount');
        }

        // Mobile transactions — approved
        if (Schema::hasTable('mobile_transactions')) {
            $sum += (float) DB::table('mobile_transactions')
                ->where('user_id', $userId)
                ->where('status', 'approved')
                ->sum('amount');
        }

        // Bank transfers — approved
        if (Schema::hasTable('bank_transfers')) {
            $sum += (float) DB::table('bank_transfers')
                ->where('user_id', $userId)
                ->where('status', 'approved')
                ->sum('amount');
        }

        // P2P Transfers — succeeded (incoming + outgoing)
        if (Schema::hasTable('transfers')) {
            $sum += (float) DB::table('transfers')
                ->where('from_user_id', $userId)
                ->where('status', 'succeeded')
                ->sum('amount');

            $sum += (float) DB::table('transfers')
                ->where('to_user_id', $userId)
                ->where('status', 'succeeded')
                ->sum('amount');
        }

        // Loan repayments — paid
        if (Schema::hasTable('loan_repayments')) {
            $sum += (float) DB::table('loan_repayments')
                ->join('loan_applications', 'loan_applications.id', '=', 'loan_repayments.loan_application_id')
                ->where('loan_applications.user_id', $userId)
                ->where('loan_repayments.status', 'paid')
                ->sum('loan_repayments.amount');
        }

        return round($sum, 2);
    }
}
