<?php

namespace App\Http\Controllers;

use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class HistoryController extends Controller
{
    public function index(Request $request)
    {
        $user = $request->user();

        // লগইন না থাকলে সেফটি
        if (! $user) {
            abort(401);
        }

        // Filters: q (search), type (credit|debit|all), kind (category)
        $type = (string) $request->query('type', '');   // credit|debit|''  -> string() / toString ঝামেলা নেই
        $kind = (string) $request->query('kind', '');   // deposits|transfers|...
        $q    = trim((string) $request->query('q', ''));

        $tx = Transaction::with('related')
            ->where('user_id', $user->id)

            // type filter
            ->when(in_array($type, ['credit', 'debit'], true), function ($qq) use ($type) {
                $qq->where('type', $type);
            })

            // kind filter (model map)
            ->when($kind !== '' && $kind !== 'all', function ($qq) use ($kind) {
                $map = [
                    'deposits'  => \App\Models\Deposit::class,
                    'transfers' => \App\Models\Transfer::class,
                    'purchases' => \App\Models\Purchase::class,
                    'recharges' => \App\Models\Recharge::class,
                    'bills'     => \App\Models\BillPayment::class,
                    'rewards'   => \App\Models\RewardConversion::class,
                    'mobile'    => \App\Models\MobileTransaction::class,
                    'bank'      => \App\Models\BankTransfer::class,
                    'loans'     => \App\Models\LoanApplication::class,
                ];

                if (array_key_exists($kind, $map)) {
                    $qq->where('related_type', $map[$kind]);
                }
            })

            // হালকা সার্চ: amount / meta / related_type
            ->when($q !== '', function ($qq) use ($q) {
                $term = '%' . $q . '%';

                $qq->where(function ($w) use ($term, $q) {
                    $w->where('amount', 'like', $term)
                      ->orWhere('meta', 'like', $term)
                      ->orWhere('related_type', 'like', '%' . Str::ucfirst($q) . '%');
                });
            })

            ->latest()
            ->paginate(20)
            ->withQueryString(); // অথবা ->appends($request->only('q','type','kind'));

        // ভিউতে দরকারি ট্যাব/ফিল্টার অপশন
        $kinds = [
            'all'       => 'সকল',
            'deposits'  => 'ডিপোজিট',
            'transfers' => 'ট্রান্সফার',
            'purchases' => 'অফার',
            'recharges' => 'রিচার্জ',
            'bills'     => 'বিল',
            'rewards'   => 'রিওয়ার্ড',
            'mobile'    => 'মোবাইল',
            'bank'      => 'ব্যাংক',
            'loans'     => 'লোন',
        ];

        return view('frontend.history.index', [
            'tx'    => $tx,
            'q'     => $q,
            'type'  => $type,
            'kind'  => $kind,
            'kinds' => $kinds,
        ]);
    }
}
