<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\LoginRequest;
use App\Http\Requests\OtpVerifyRequest;
use App\Mail\LoginOtpMail;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class LoginController extends Controller
{
    /**
     * Step 1: Login form
     */
     public function redirectDash()
{
    $redirect = '';

    if (Auth::check() && Auth::user()->role == 'user') {
        $redirect = '/dashboard';
    }
    else if (Auth::check() && Auth::user()->role == 'admin') {
        $redirect = '/admin/dashboard';
    }
    else {
        $redirect = '/';
    }

    return $redirect;
}

    public function showLogin()
    {
        if (Auth::check()) {
    return redirect($this->redirectDash());
}

        return view('frontend.auth.login');
    }

    /**
     * Step 1: email + password verify, send OTP
     */
    public function attemptLogin(LoginRequest $request)
    {
        $user = User::where('email', $request->email)->first();

        $valid = false;

        if ($user) {
            if ($user->isBlocked()) {
                throw ValidationException::withMessages([
                    'email' => ['আপনার একাউন্ট ব্লক করা আছে। এডমিনের সাথে যোগাযোগ করুন।'],
                ]);
            }

            try {
                $valid = Hash::check($request->password, $user->password);
            } catch (\RuntimeException $e) {
                $valid = $this->checkAndMigratePassword($user, $request->password);
            }

            if (! $valid) {
                $valid = $this->checkAndMigratePassword($user, $request->password);
            }
        }

        if (! $user || ! $valid) {
            throw ValidationException::withMessages([
                'email' => ['ইমেইল বা পাসওয়ার্ড সঠিক নয়।'],
            ]);
        }

        // ----- Create OTP (6-digit, 10 minutes) -----
        $plain = str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $hash  = Hash::make($plain);
        $ttl   = 10;

        DB::table('login_otps')->insert([
            'user_id'    => $user->id,
            'code_hash'  => $hash,
            'sent_to'    => $user->email,
            'expires_at' => Carbon::now()->addMinutes($ttl),
            'ip'         => $request->ip(),
            'user_agent' => substr((string) $request->userAgent(), 0, 255),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        Mail::to($user->email)->send(new LoginOtpMail($plain, $ttl));

        // session state for pending login
        session([
            'login.pending_user_id' => $user->id,
            'login.remember'        => true, // সবসময় remember => ১+ বছর
        ]);

        return redirect()
            ->route('otp.verify.form')
            ->with('status', 'OTP পাঠানো হয়েছে। আপনার ইমেইল চেক করুন।');
    }

    /**
     * Legacy / mismatch password hash verify + migrate
     */
    private function checkAndMigratePassword(User $user, string $plain): bool
    {
        $stored = (string) $user->password;

        $info = password_get_info($stored);
        if (! empty($info['algo'])) {
            if (password_verify($plain, $stored)) {
                if (Hash::needsRehash($stored)) {
                    $user->forceFill(['password' => Hash::make($plain)])->save();
                }

                return true;
            }

            return false;
        }

        // ধরে নিচ্ছি plain text stored ছিল
        if (hash_equals($stored, $plain)) {
            $user->forceFill(['password' => Hash::make($plain)])->save();

            return true;
        }

        return false;
    }

    /**
     * Step 2: OTP form
     */
    public function showOtpForm()
    {
        if (! session()->has('login.pending_user_id')) {
            return redirect()->route('login')
                ->withErrors(['email' => 'প্রথমে ইমেইল ও পাসওয়ার্ড দিন।']);
        }

        return view('frontend.auth.verify-otp');
    }

    /**
     * Step 2: OTP verify + login
     */
    public function verifyOtp(OtpVerifyRequest $request)
    {
        $pendingUserId = session('login.pending_user_id');
        if (! $pendingUserId) {
            return redirect()->route('login')
                ->withErrors(['email' => 'সেশনটির মেয়াদ শেষ হয়েছে। আবার চেষ্টা করুন।']);
        }

        $otpRow = DB::table('login_otps')
            ->where('user_id', $pendingUserId)
            ->whereNull('consumed_at')
            ->where('expires_at', '>', now())
            ->latest('id')
            ->first();

        if (! $otpRow) {
            throw ValidationException::withMessages([
                'otp' => ['OTP পাওয়া যায়নি বা মেয়াদ শেষ। আবার পাঠিয়ে নিন।'],
            ]);
        }

        DB::table('login_otps')->where('id', $otpRow->id)->increment('attempts');

        if (! Hash::check($request->otp, $otpRow->code_hash)) {
            throw ValidationException::withMessages([
                'otp' => ['OTP সঠিক নয়।'],
            ]);
        }

        DB::table('login_otps')
            ->where('id', $otpRow->id)
            ->update([
                'consumed_at' => now(),
                'updated_at'  => now(),
            ]);

        // Remember = true => দীর্ঘমেয়াদী login
        Auth::loginUsingId($pendingUserId, true);

        /** @var User $user */
        $user = Auth::user();

        // ✅ একাউন্ট এক ডিভাইস চেক – কিন্তু শুধু user role এর জন্য
        $this->checkUserDeviceOrFail($request, $user);

        // session clean
        session()->forget(['login.pending_user_id', 'login.remember']);

        // নতুন সেশনে প্রথমে PIN চাই – verified false
        session(['pin.verified' => false]);

        return redirect()
            ->route('pin.form')
            ->with('success', 'লগইন সফল হয়েছে, এখন PIN দিন।');
    }

    /**
     * একাউন্ট এক ডিভাইস:
     *
     * ✅ এখন শুধু user role এর জন্য সীমাবদ্ধ
     *
     * - প্রথমবার login → user.device_token null ⇒ ডিভাইস bind করে দেই, device_lock = true
     * - একই ডিভাইস (cookie token == user.device_token) ⇒ OK, শুধু cookie refresh
     * - অন্য ডিভাইস / cookie নাই ⇒
     *      - যদি device_lock = true ⇒ ব্লক করে দেই (admin permission লাগবে)
     *      - যদি device_lock = false ⇒ একবারের জন্য allow, নতুন token bind,
     *        আবার device_lock = true করে দেই
     *
     * 🔹 admin role এখানে একদমই চেক হবে না – যত ডিভাইস থেকে খুশি লগইন করতে পারবে।
     */
    private function checkUserDeviceOrFail(Request $request, User $user): void
    {
        // ✅ admin হলে: device lock apply হবে না, সরাসরি return
        if ($user->role === 'admin') {
            return;
        }

        $cookieName  = 'device_token';
        $cookieToken = (string) $request->cookie($cookieName);
        $userToken   = (string) $user->device_token;

        // একাউন্ট ব্লক হলে
        if ($user->isBlocked()) {
            Auth::logout();

            throw ValidationException::withMessages([
                'email' => ['আপনার একাউন্ট ব্লক করা আছে। এডমিনের সাথে যোগাযোগ করুন।'],
            ]);
        }

        // 1) প্রথমবার login – এখনো কোনো device_token নেই
        if (empty($userToken)) {
            $token = Str::random(60);

            $user->forceFill([
                'device_token' => $token,
                // ডিফল্টে লক করে রাখব – একটায়ই login
                'device_lock'  => true,
            ])->save();

            cookie()->queue(
                cookie($cookieName, $token, 60 * 24 * 365) // ১ বছর
            );

            return;
        }

        // 2) একই ডিভাইস – cookie token এবং user token match করে
        if (! empty($cookieToken) && hash_equals($userToken, $cookieToken)) {
            // same device ⇒ শুধু cookie refresh
            cookie()->queue(
                cookie($cookieName, $userToken, 60 * 24 * 365)
            );

            return;
        }

        // 3) এখানে মানে:
        //   - userToken আছে
        //   - কিন্তু cookie নাই (clear করা হয়েছে / নতুন ব্রাউজার)
        //   - অথবা cookie আছে কিন্তু mismatch (নতুন ডিভাইস)

        // যদি device_lock = true ⇒ নতুন ডিভাইসে allow করব না
        if ((bool) $user->device_lock === true) {
            Auth::logout();

            throw ValidationException::withMessages([
                'email' => [
                    'আপনার একাউন্ট একটি ডিভাইসে লক করা আছে। নতুন ডিভাইসে লগইন করতে এডমিনের অনুমতি প্রয়োজন।',
                ],
            ]);
        }

        // 4) এখানে এডমিন আগে থেকে device_lock = false করে দিয়েছেন
        // এখন একবারের জন্য নতুন ডিভাইস allow করব,
        // নতুন token bind হবে, আবার device_lock = true
        $newToken = Str::random(60);

        $user->forceFill([
            'device_token' => $newToken,
            'device_lock'  => true, // আবার লক করে দিলাম
        ])->save();

        cookie()->queue(
            cookie($cookieName, $newToken, 60 * 24 * 365)
        );
    }

    /**
     * OTP resend
     */
    public function resendOtp(Request $request)
    {
        $pendingUserId = session('login.pending_user_id');
        if (! $pendingUserId) {
            return redirect()->route('login')
                ->withErrors(['email' => 'সেশনটির মেয়াদ শেষ হয়েছে। আবার চেষ্টা করুন।']);
        }

        $user = User::find($pendingUserId);
        if (! $user || ! $user->email) {
            return redirect()->route('login')
                ->withErrors(['email' => 'ইউজার পাওয়া যায়নি।']);
        }

        $plain = str_pad((string) random_int(0, 999999), 6, '0', STR_PAD_LEFT);
        $hash  = Hash::make($plain);
        $ttl   = 10;

        DB::table('login_otps')->insert([
            'user_id'    => $user->id,
            'code_hash'  => $hash,
            'sent_to'    => $user->email,
            'expires_at' => now()->addMinutes($ttl),
            'ip'         => $request->ip(),
            'user_agent' => substr((string) $request->userAgent(), 0, 255),
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        Mail::to($user->email)->send(new LoginOtpMail($plain, $ttl));

        return back()->with('status', 'নতুন OTP পাঠানো হয়েছে। ইমেইল চেক করুন।');
    }

    /**
     * Logout
     */
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        // PIN verify flag remove
        $request->session()->forget('pin.verified');

        return redirect()->route('login')
            ->with('status', 'আপনি লগআউট হয়েছেন।');
    }
}
