<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\MobileTransaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class MobileTransactionController extends Controller
{
    public function index(Request $request) {
        $q = MobileTransaction::with(['user','gateway'])->latest();
        if ($request->filled('status')) $q->where('status', $request->string('status'));
        $transactions = $q->paginate(20);
        return view('admin.transactions.index', compact('transactions'));
    }

    public function approve(MobileTransaction $transaction, Request $request) {
        // pending না হলে ব্লক করুন
        if ($transaction->status !== 'pending') {
            return back()->with('info', 'শুধু pending ট্রানজ্যাকশন অ্যাপ্রুভ করা যায়।');
        }

        try {
            DB::transaction(function () use ($transaction, $request) {
                // ইউজারকে লক করে নিই—রেস কন্ডিশন এড়াতে
                $user = $transaction->user()->lockForUpdate()->first();

                $amount = (float) $transaction->amount;

                // ব্যালান্স চেক
                if ((float)$user->balance < $amount) {
                    // ভ্যালিডেশন ফেল করলে রোলব্যাক হবে
                    abort(400, 'ইউজারের পর্যাপ্ত ব্যালান্স নেই—অ্যাপ্রুভ করা গেল না।');
                }

                // 1) ব্যালান্স ডেবিট
                $user->balance = round(((float)$user->balance - $amount), 2);

                // 2) কমিশন রিওয়ার্ড পয়েন্টে যোগ (ইন্টিজার হিসেবে)
                $user->reward_points = (int) $user->reward_points + (int) round($transaction->commission_amount);

                $user->save();

                // 3) ট্রানজ্যাকশন আপডেট
                $transaction->update([
                    'status'     => 'approved',
                    'admin_note' => $request->string('admin_note'),
                ]);
            });

            return back()->with('success', 'ট্রানজ্যাকশন অ্যাপ্রুভ হয়েছে, ইউজারের ব্যালান্স থেকে টাকা কাটা হয়েছে এবং কমিশন যোগ হয়েছে।');
        } catch (\Throwable $e) {
            // ব্যালান্স না থাকা বা অন্য কোনো ইস্যু
            return back()->with('error', $e->getMessage());
        }
    }

    public function reject(MobileTransaction $transaction, Request $request) {
        if ($transaction->status !== 'pending') {
            return back()->with('info', 'শুধু pending ট্রানজ্যাকশন রিজেক্ট করা যায়।');
        }
        $transaction->update([
            'status'     => 'rejected',
            'admin_note' => $request->string('admin_note'),
        ]);
        return back()->with('success', 'ট্রানজ্যাকশন রিজেক্ট করা হয়েছে।');
    }
}
