<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;


class DepositController extends Controller
{
    // public function __construct() { $this->middleware(['auth','can:admin']); }

    public function index(Request $request)
    {
        $status   = $request->get('status');
        $query    = Deposit::with(['user','method'])->latest();
        if ($status) $query->where('status', $status);
        $deposits = $query->paginate(30)->withQueryString();

        return view('admin.deposits.index', compact('deposits','status'));
    }

    public function show(Deposit $deposit)
    {
        $deposit->load(['user','method']);
        return view('admin.deposits.show', compact('deposit'));
    }

    public function approve(Deposit $deposit, Request $request)
    {
        abort_if($deposit->status !== 'pending', 400, 'Already processed.');

        DB::transaction(function () use ($deposit, $request) {
            $user = $deposit->user()->lockForUpdate()->first();

            // credit wallet
            $user->balance = $user->balance + $deposit->amount;
            $user->save();

            // mark approved
            $deposit->update([
                'status'      => 'approved',
                'approved_at' => now(),
                'admin_note'  => $request->input('note'),
            ]);

            // transaction log
            Transaction::create([
                'user_id'       => $user->id,
                'type'          => 'credit',
                'amount'        => $deposit->amount,
                'balance_after' => $user->balance,
                'related_type'  => \App\Models\Deposit::class,
                'related_id'    => $deposit->id,
                'meta'          => ['reference'=>$deposit->reference],
            ]);
        });

    
        return back()->with('ok','Deposit approved.');
    }

    public function reject(Deposit $deposit, Request $request)
    {
        abort_if($deposit->status !== 'pending', 400, 'Already processed.');
        $data = $request->validate(['note'=>['required','string','max:500']]);
        $deposit->update(['status'=>'rejected','admin_note'=>$data['note']]);
   
        return back()->with('ok','Deposit rejected.');
    }
}
