<?php

namespace App\Http\Controllers;

use App\Models\AddmoneyBankMethod as AddMoneyBankMethod;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class AddMoneyBankMethodController extends Controller
{
    public function index(Request $request)
    {
        $q = AddMoneyBankMethod::query();

        if ($search = $request->input('q')) {
            $q->where(function ($qq) use ($search) {
                $qq->where('bank_name', 'like', "%{$search}%")
                   ->orWhere('slug', 'like', "%{$search}%")
                   ->orWhere('account_number', 'like', "%{$search}%")
                   ->orWhere('account_holder', 'like', "%{$search}%");
            });
        }

        $methods = $q->orderBy('display_order')->orderBy('bank_name')->paginate(12)->withQueryString();

        return view('admin.bank_methods.index', compact('methods'));
    }

    public function create()
    {
        return view('admin.bank_methods.create');
    }

    public function store(Request $request)
    {
        $data = $this->validated($request);

        // slug: if not provided, derive from bank_name
        $data['slug'] = Str::slug($data['slug'] ?? $data['bank_name']);

        // logo upload
        if ($request->hasFile('bank_logo')) {
            $data['bank_logo'] = $request->file('bank_logo')->store('bank-logos', 'public');
        }

        // meta (optional JSON text field)
        if ($request->filled('meta_raw')) {
            $data['meta'] = json_decode($request->meta_raw, true) ?? null;
        }

        AddMoneyBankMethod::create($data);

        return redirect()->route('bank_methods.index')->with('success', 'Bank method created.');
    }

    public function edit($id)
    {
        $method = AddMoneyBankMethod::findOrFail($id);
        return view('admin.bank_methods.edit', compact('method'));
    }

    public function update(Request $request, $id)
    {
        $method = AddMoneyBankMethod::findOrFail($id);

        $data = $this->validated($request, $method->id);

        // slug: if blank, keep previous or re-derive
        $data['slug'] = Str::slug($data['slug'] ?? $method->slug ?? $data['bank_name']);

        // logo upload (replace old if new provided)
        if ($request->hasFile('bank_logo')) {
            if ($method->bank_logo && Storage::disk('public')->exists($method->bank_logo)) {
                Storage::disk('public')->delete($method->bank_logo);
            }
            $data['bank_logo'] = $request->file('bank_logo')->store('bank-logos', 'public');
        }

        // meta from raw JSON textarea
        if ($request->filled('meta_raw')) {
            $data['meta'] = json_decode($request->meta_raw, true) ?? null;
        } elseif ($request->has('meta_raw') && $request->meta_raw === '') {
            $data['meta'] = null;
        }

        $method->update($data);

        return redirect()->route('bank_methods.index')->with('success', 'Bank method updated.');
    }

    public function destroy($id)
    {
        $method = AddMoneyBankMethod::findOrFail($id);

        // (optional) delete logo file on soft delete? usually no. Uncomment if you want hard delete behavior.
        // if ($method->bank_logo && Storage::disk('public')->exists($method->bank_logo)) {
        //     Storage::disk('public')->delete($method->bank_logo);
        // }

        $method->delete();

        return redirect()->route('bank_methods.index')->with('success', 'Bank method deleted.');
    }

    public function toggle($id)
    {
        $method = AddMoneyBankMethod::findOrFail($id);
        $method->is_active = ! $method->is_active;
        $method->save();

        return back()->with('success', 'Status toggled.');
    }

    private function validated(Request $request, $ignoreId = null): array
    {
        return $request->validate([
            'bank_name'       => ['required','string','max:100'],
            'bank_logo'       => ['nullable','image','mimes:jpg,jpeg,png,webp','max:2048'],
            'slug'            => [
                'nullable','string','max:100',
                Rule::unique('addmoney_bank_methods','slug')->ignore($ignoreId),
            ],
            'account_number'  => ['required','string','max:64'],
            'account_holder'  => ['required','string','max:100'],
            'location'        => ['nullable','string','max:191'],
            'daily_min'       => ['nullable','numeric','min:0'],
            'daily_max'       => ['nullable','numeric','min:0'],
            'is_active'       => ['sometimes','boolean'],
            'display_order'   => ['nullable','integer','min:0'],
            // meta: accept textarea raw JSON
            'meta_raw'        => ['nullable','string'],
        ], [
            'slug.unique'     => 'The slug must be unique.',
        ]);
    }
}
