<?php

namespace App\Http\Controllers;

use App\Models\AddmoneyBank as AddMoneyBank;
use App\Models\AddmoneyBankMethod as AddMoneyBankMethod;
use App\Models\Transaction;
use App\Models\BellNotification;         // ✅ Notification model import
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Events\BankAddMoney;             // ✅ Event import
use Illuminate\Support\Facades\Auth;     // ✅ User
use Illuminate\Support\Facades\Hash;     // ✅ login_pin check

class AddMoneyBankController extends Controller
{
    public function methods()
    {
        $methods = AddMoneyBankMethod::where('is_active', true)
            ->orderBy('display_order')
            ->orderBy('bank_name')
            ->get();

        return view('frontend.add-money-bank.methods', compact('methods'));
    }

    public function create(string $slug)
    {
        $method = AddMoneyBankMethod::where('slug', $slug)->firstOrFail();
        return view('frontend.add-money-bank.create', compact('method'));
    }

    public function store(Request $request, string $slug)
    {
        $method = AddMoneyBankMethod::where('slug', $slug)->firstOrFail();

        // 🔐 form validation (login_pin সহ)
        $data = $request->validate([
            'sender_branch'    => ['nullable', 'string'],
            'routing_number'   => ['nullable', 'string'],
            'account_number'   => ['required', 'string'],
            'account_holder'   => ['required', 'string'],
            'location'         => ['nullable', 'string'],
            'amount'           => ['required', 'numeric'],
            'proof_screenshot' => ['nullable', 'file', 'mimes:jpg,jpeg,png,pdf'],

            // ✅ লগইন PIN ৪ digit
            'login_pin'        => ['required', 'digits:4'],
        ]);

        // 1️⃣ current user
        $user = Auth::user();
        if (! $user) {
            abort(401);
        }

        // ✅ user এর login_pin সেট করা আছে কি না
        if (empty($user->login_pin)) {
            return back()
                ->withErrors([
                    'login_pin' => 'আপনি এখনো লগইন PIN সেট করেননি। আগে প্রোফাইল থেকে PIN সেট করে নিন।',
                ])
                ->withInput();
        }

        // ✅ login_pin মিলাচ্ছি
        if (! Hash::check($data['login_pin'], $user->login_pin)) {
            return back()
                ->withErrors([
                    'login_pin' => 'ভুল লগইন PIN দেওয়া হয়েছে।',
                ])
                ->withInput();
        }

        // এরপর আর login_pin দরকার নেই
        unset($data['login_pin']);

        // 2️⃣ limit check
        if (! is_null($method->daily_min) && $data['amount'] < $method->daily_min) {
            return back()
                ->withErrors(['amount' => "Minimum amount is {$method->daily_min}."])
                ->withInput();
        }

        if (! is_null($method->daily_max) && $data['amount'] > $method->daily_max) {
            return back()
                ->withErrors(['amount' => "Maximum amount is {$method->daily_max}."])
                ->withInput();
        }

        // 3️⃣ file upload
        $path = $request->hasFile('proof_screenshot')
            ? $request->file('proof_screenshot')->store('bank-deposits', 'public')
            : null;

        // 4️⃣ transaction + bank deposit একসাথে
        $bank = DB::transaction(function () use ($request, $method, $data, $path, $user) {

            // ✅ Bank deposit create
            $bank = AddMoneyBank::create([
                'user_id'                 => $user->id,
                'addmoney_bank_method_id' => $method->id,
                'sender_branch'           => $data['sender_branch'] ?? null,
                'routing_number'          => $data['routing_number'] ?? null,
                'account_number'          => $data['account_number'],
                'account_holder'          => $data['account_holder'],
                'location'                => $data['location'] ?? null,
                'amount'                  => $data['amount'],
                'reference'               => Str::upper(Str::random(10)),
                'proof_screenshot'        => $path,
                'status'                  => 'pending',
                'ip'                      => $request->ip(),
                'user_agent'              => substr((string) $request->header('User-Agent'), 0, 255),
            ]);

            // ✅ Transaction create (polymorphic, balance pending অবস্থায় change করছ না)
            $txn = new Transaction([
                'user_id'       => $user->id,
                'type'          => 'credit',
                'amount'        => $data['amount'],
                'balance_after' => $user->balance,  // current balance reference
                'meta'          => [
                    'status'    => 'pending',
                    'reference' => $bank->reference,
                    'method'    => $method->slug,
                    'note'      => 'Bank deposit submitted; awaiting review',
                ],
            ]);

            $txn->related()->associate($bank);
            $txn->saveOrFail();

            // ✅ Bell Notification create
            $bankName = $method->bank_name ?? $method->slug;

            BellNotification::create([
                'user_id' => $user->id,
                'title'   => 'New Bank Deposit Submitted',
                'status'  => 'unread',
                'type'    => 'bank_deposit',
                'amount'  => $data['amount'],
                'message' =>
                    "Your bank deposit of {$data['amount']} TK via {$bankName} ".
                    "is submitted and waiting for approval. Reference: {$bank->reference}",
                'read_at' => null,
            ]);

            // ✅ Event fire → broadcast, ইত্যাদি
            event(new BankAddMoney($bank));

            return $bank;
        }, 3);

        return redirect()
            ->route('addmoney_bank.success', $bank)
            ->with('success', 'Bank deposit submitted. Awaiting review.');
    }

    public function success(AddMoneyBank $bank)
    {
        return view('frontend.add-money-bank.success', compact('bank'));
    }

    public function transactions()
    {
        $banks = AddMoneyBank::where('user_id', auth()->id())
            ->latest()
            ->paginate(20);

        return view('frontend.add-money-bank.transactions', compact('banks'));
    }
}
